/***********************************************************
 * $Id: ConfigurationService.java 374 2018-10-27 21:07:11Z login $
 * 
 * http://www.clazzes.org
 *
 * Created: 02.04.2011
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.external;

import java.util.ArrayList;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.Vector;

import org.clazzes.util.http.sec.HttpLoginService;
import org.clazzes.util.sched.IOneTimeScheduler;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService implements ManagedService {

    private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);

    private static final long DEFAULT_REFRESH_CRL_MINUTES = 1440;
    private static final long DEFAULT_REFRESH_CRL_MINUTES_ON_FAILURE = 5;

    /**
     * A mapping from issuer DNs to issuer configs.
     */
    private final Map<String,IssuerConfig> issuers;
    /**
     * A mapping from a context name to a whitelist of subject DNs
     */
    private final Map<String,Set<String>> contexts;

    private IOneTimeScheduler oneTimeScheduler;
    private final List<UUID> jobIds;
    
    public ConfigurationService() {
        this.issuers = new HashMap<String,IssuerConfig>();
        this.contexts = new HashMap<String, Set<String>>();
        this.jobIds = new ArrayList<UUID>();
    }
    
    private void stopJobs() {
        
        if (this.jobIds != null) {
            
            for (UUID jobId : this.jobIds) {
                log.info("Cancelling CRL update job with ID [{}].",jobId);
                this.oneTimeScheduler.cancelJob(jobId,true);
            }
        }
    }
    
    
    public synchronized void destroy()  {

        this.issuers.clear();
        this.contexts.clear();
        
        this.stopJobs();
    }

    @Override
    public synchronized void updated(Dictionary<String,?> properties) throws ConfigurationException {

        this.issuers.clear();
        this.contexts.clear();
        
        this.stopJobs();
        
        this.jobIds.clear();
        
        if (properties != null) {
            
            long refreshCrlMinutes = DEFAULT_REFRESH_CRL_MINUTES;
            
            Object refreshCrlMinutes_o = properties.get("refreshCrlMinutes");
            
            if (refreshCrlMinutes_o != null) {
                
                refreshCrlMinutes = Long.parseLong(refreshCrlMinutes_o.toString());
            }
            
            long refreshCrlMinutesOnFailure = DEFAULT_REFRESH_CRL_MINUTES_ON_FAILURE;

            Object refreshCrlMinutesOnFailure_o = properties.get("refreshCrlMinutesOnFailure");
            
            if (refreshCrlMinutesOnFailure_o != null) {
                
                refreshCrlMinutesOnFailure = Long.parseLong(refreshCrlMinutesOnFailure_o.toString());
            }
            
            log.info("Using refreshCrlMinutes [{}].",refreshCrlMinutes);
            log.info("Using refreshCrlMinutesOnFailure [{}].",refreshCrlMinutesOnFailure);
            
            Enumeration<String> keys = properties.keys();

            Map<String,Set<String>> blacklists = new HashMap<String,Set<String>>();
            
            while (keys.hasMoreElements()) {

                String key = keys.nextElement();

                if (key.startsWith("whitelist.")) {

                    int idx = key.lastIndexOf('.');
                    
                    if (idx > 10) {
                        
                        String ctxt = key.substring(10,idx);
                        
                        Set<String> whitelist = this.contexts.get(ctxt);
                        
                        if (whitelist == null) {
                            whitelist = new HashSet<String>();
                            this.contexts.put(ctxt,whitelist);
                        }
                        
                        String wdn = properties.get(key).toString();
                        
                        log.info("Adding [{}] to whitelist for context [{}]",wdn,ctxt);
                        
                        whitelist.add(wdn);
                    }
                }
                        
                if (key.startsWith("blacklist.")) {

                    int idx = key.lastIndexOf('.');
                    
                    if (idx > 10) {
                        
                        String label = key.substring(10,idx);
                        
                        Set<String> blacklist = blacklists.get(label);
                        
                        if (blacklist == null) {
                            blacklist = new HashSet<String>();
                            blacklists.put(label,blacklist);
                        }
                        
                        String bdn = properties.get(key).toString();
                        
                        log.info("Adding [{}] to blacklist for issuer [{}]",bdn,label);
                        
                        blacklist.add(bdn);
                    }
                    
                }

                if (key.startsWith("issuer.") && key.endsWith(".dn")) {

                    String label = key.substring(7,key.length()-3);
                    String dn = properties.get(key).toString();

                    Set<String> blacklist = blacklists.get(label);
                    
                    if (blacklist == null) {
                        blacklist = new HashSet<String>();
                        blacklists.put(label,blacklist);
                    }
                    
                    log.info("Setting DN for issuer [{}] to [{}].",label,dn);
                    
                    Object crlDistributionPoint_o = properties.get("issuer."+label+".crlDistributionPoint");
                    
                    String crlDistributionPoint = null;
                    
                    if (crlDistributionPoint_o != null) {
                        
                        log.info("Setting CRL distribution point for issuer [{}] to [{}].",label,crlDistributionPoint_o);
                        
                        crlDistributionPoint = crlDistributionPoint_o.toString();
                    }
                    
                    Object crlVerificationCertificate_o = properties.get("issuer."+label+".crlVerificationCertificate");
                    
                    String crlVerificationCertificate = null;
                    
                    if (crlVerificationCertificate_o != null) {
                        
                        log.info("Setting CRL verification certificate for issuer [{}] to [{}].",label,crlVerificationCertificate_o);
                        
                        crlVerificationCertificate = crlVerificationCertificate_o.toString();
                    }

                    this.issuers.put(dn,new IssuerConfig(crlDistributionPoint,
                            crlVerificationCertificate,blacklist,refreshCrlMinutes,refreshCrlMinutesOnFailure));
                }
            }
        }
        
        for (Entry<String, IssuerConfig> e : this.issuers.entrySet()) {
            
            if (e.getValue().getCrlDistributionPoint() != null) {
            
                UUID jobId = this.oneTimeScheduler.scheduleJob(e.getValue());
                
                log.info("Started CRL background job for issuer [{}] with job ID [{}]",e.getKey(),jobId);
                this.jobIds.add(jobId);
            }
        }
    }

    /**
     * @return the list of issuer DNS for which we have an issuer config.
     */
    public synchronized List<String> getIssuerDns() {
        Vector<String> domains = new Vector<String>(this.issuers.size());
        domains.addAll(this.issuers.keySet());
        return domains;
    }

    /**
     * @param issuerDn A configured issuer DN.
     * @return The domain controller URI for the given domain.
     */
    public synchronized IssuerConfig getIssuerConfig(String issuerDn) {

        return this.issuers.get(issuerDn);
    }

    /**
     * Return a white list for the configured context.
     * 
     * @param context A context to be used as suffix like
     *          <code>/ext-login/org.clazzes.login.x509/ctxt/{context}</code> in
     *          {@link HttpLoginService#checkPermission(javax.servlet.http.HttpServletRequest, String)}
     * @return A set of whitelisted subject DNs for the given context.
     */
    public synchronized Set<String> getWhiteList(String context) {
        return this.contexts.get(context);
    }

    /**
     * @param oneTimeScheduler The one time scheduler to set.
     */
    public void setOneTimeScheduler(IOneTimeScheduler oneTimeScheduler) {
        this.oneTimeScheduler = oneTimeScheduler;
    }
    
}
