/***********************************************************
 * $Id$
 * 
 * Multi backend broker variant of clazzes.org Login Service API
 * 
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.login.broker;

import java.util.Dictionary;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.clazzes.util.osgi.ServiceMapListener;
import org.clazzes.util.sec.DomainPasswordLoginService;
import org.osgi.service.cm.ConfigurationException;
import org.osgi.service.cm.ManagedService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConfigurationService implements ServiceMapListener, ManagedService {

  private static final Logger log = LoggerFactory.getLogger(ConfigurationService.class);
  
  private String defaultDomain;
  
  private Map<String, DomainPasswordLoginService> loginServicesByDomain;
  private Map<String, DomainPasswordLoginService> loginServicesByMechanism;


  public ConfigurationService() {
    this.defaultDomain = null;
    this.loginServicesByDomain = new HashMap<String, DomainPasswordLoginService>();
    this.loginServicesByMechanism = new HashMap<String, DomainPasswordLoginService>();
  }
  
  
  @SuppressWarnings("rawtypes")
  @Override
  public synchronized void updated(Dictionary properties) throws ConfigurationException {

    Object s = properties == null ? null : properties.get("defaultDomain");
    if (s != null) {
      this.defaultDomain = s.toString();
      if (log.isDebugEnabled())
        log.debug("Setting default domain to [{}].",this.defaultDomain);
    }
    else {
      this.defaultDomain = null;
      if (log.isDebugEnabled())
        log.debug("Setting default domain to default [{}].",this.defaultDomain);
    }

    // FUTURE: Support exclicit domain configuration?
    
  }
  

  public synchronized String getDefaultDomain() {
    return defaultDomain;
  }


  public synchronized void setDefaultDomain(String defaultDomain) {
    this.defaultDomain = defaultDomain;
  }


  @Override
  public synchronized void serviceBound(String mechanism, Object service) {
    
    if ("org.clazzes.login.broker".equals(mechanism)) {
      log.info("Not registering myself, with mechanism [{}]", mechanism);
      return; // avoid recursion!
    }
    
    DomainPasswordLoginService domainPasswordLoginService = (DomainPasswordLoginService)service;
    
    log.info("Remembering DomainPasswordLoginService for mechanism [{}]", mechanism);
    this.loginServicesByMechanism.put(mechanism, domainPasswordLoginService);
    
    HashSet<String> dplsDomains = new HashSet<String>();
    
    String dplsDefaultDomain = domainPasswordLoginService.getDefaultDomain();
    if (dplsDefaultDomain != null && dplsDefaultDomain.length() > 0) {
      dplsDomains.add(dplsDefaultDomain);
    } else {
      log.warn("DomainPasswordLoginService with mechanism [{}] does not have a default domain.", mechanism);
    }
    
    List<String> domains = domainPasswordLoginService.getDomains();
    
    if (domains != null) {
  	  dplsDomains.addAll(domains);
    }
    
    if (dplsDomains.size() == 0) {
      log.warn("DomainPasswordLoginService with mechanism [{}] does not cover any domain. Ignoring it.", mechanism);
      return;
    }
    
    for (String dplsDomain : dplsDomains) {
      DomainPasswordLoginService knownDpls = this.loginServicesByDomain.get(dplsDomain);
      if (knownDpls == null) {
        log.info("Remebering DomainPasswordLoginService for mechanism [{}] to cover domain [{}]", mechanism, dplsDomain);
        this.loginServicesByDomain.put(dplsDomain, domainPasswordLoginService);
      } else {
        log.error("A DomainPasswordLoginService for mechanism [{}] covering domain [{}] is already present, ignoring new one.", mechanism, dplsDomain);
        // throw Exception?
      }
    }
  }


  @Override
  public synchronized void serviceUnbound(String mechanism, Object service) {

    if ("org.clazzes.login.broker".equals(mechanism)) {
      log.info("Not unregistering myself, with mechanism [{}]", mechanism);
      return; // avoid recursion!
    }
    
    DomainPasswordLoginService domainPasswordLoginService = (DomainPasswordLoginService)service;

    
    log.info("Forgetting about DomainPasswordLoginService for mechanism [{}]", mechanism);
    DomainPasswordLoginService removedDpls = this.loginServicesByMechanism.remove(mechanism);
    if (removedDpls == null) {
      log.info("Had no knowledge about DomainPasswordLoginService for mechanism [{}]", mechanism);
    }

    HashSet<String> dplsDomains = new HashSet<String>();
    
    String dplsDefaultDomain = domainPasswordLoginService.getDefaultDomain();
    if (dplsDefaultDomain != null && dplsDefaultDomain.length() > 0) {
      dplsDomains.add(dplsDefaultDomain);
    } else {
      log.warn("DomainPasswordLoginService with mechanism [{}] does not have a default domain.", mechanism);
    }
    
    List<String> domains = domainPasswordLoginService.getDomains();
    
    if (domains != null) {
  	  dplsDomains.addAll(domains);
    }
    
    if (dplsDomains.size() == 0) {
      log.warn("DomainPasswordLoginService with mechanism [{}] does not cover any domain. Ignoring it.", mechanism);
      return;
    }
    
    for (String dplsDomain : dplsDomains) {
      log.info("Forgetting about DomainPasswordLoginService for mechanism [{}] covering domain [{}]", mechanism, dplsDomain);
      removedDpls = this.loginServicesByDomain.remove(dplsDomain);
      if (removedDpls == null) {
        log.info("Had no knowledge about DomainPasswordLoginService for mechanism [{}] covering domain [{}]", mechanism, dplsDomain);
      } // evtl. check if it was the same service object?
    }
    
  }

  
  public synchronized DomainPasswordLoginService getLoginService(String domain) {
    DomainPasswordLoginService domainPasswordLoginService = this.loginServicesByDomain.get(domain);
    if (domainPasswordLoginService == null) {
      log.info("Request for DomainPasswordLoginService for domain [{}] cannot be covered from cache, searching all known services by mechanism");
      rescanLoginServices();
      domainPasswordLoginService = this.loginServicesByDomain.get(domain);
    }
    if (domainPasswordLoginService == null) {
      log.warn("Request for DomainPasswordLoginService for domain [{}] cannot be satisfied.");
    }
    return domainPasswordLoginService;
  }
  
  
  public synchronized void rescanLoginServices() {
    for (String mechanism : this.loginServicesByMechanism.keySet()) {
      log.info("Re-Scanning DomainPasswordLoginService for mechanism [{}]", mechanism);
      DomainPasswordLoginService loginService = this.loginServicesByMechanism.get(mechanism);
      
      HashSet<String> dplsDomains = new HashSet<String>();
      
      String dplsDefaultDomain = loginService.getDefaultDomain();
      if (dplsDefaultDomain != null && dplsDefaultDomain.length() > 0) {
        dplsDomains.add(dplsDefaultDomain);
      } else {
        log.warn("DomainPasswordLoginService for mechanism [{}] has no default domain", mechanism);
      }

      List<String> domains = loginService.getDomains();
      
      if (domains != null) {
    	  dplsDomains.addAll(loginService.getDomains());
      }
    		  
      if (dplsDomains.size() == 0) {
        log.warn("DomainPasswordLoginService for mechanism [{}] does not cover any domains, skipping it...", mechanism);
        continue;
      }
      
      for (String dplsDomain : dplsDomains) {
        //log.debug("DomainPasswordLoginService for mechanism [{}] covers domain [{}]", mechanism, dplsDomain);
        DomainPasswordLoginService knownDpls = this.loginServicesByDomain.get(dplsDomain);
        if (knownDpls == null) {
          log.info("Found out about known DomainPasswordLoginService for mechanism [{}] to cover the additional domain [{}]", mechanism, dplsDomain);
          this.loginServicesByDomain.put(dplsDomain, loginService);
        } else {
          log.debug("I already known about DomainPasswordLoginService for mechanism [{}] to cover the domain [{}]", mechanism, dplsDomain);
        }
      }
    }
  }


  public List<String> getDomains() {
    HashSet<String> dplsDomains = new HashSet<String>();

    for (String mechanism : this.loginServicesByMechanism.keySet()) {
      DomainPasswordLoginService loginService = this.loginServicesByMechanism.get(mechanism);

      String dplsDefaultDomain = loginService.getDefaultDomain();
      if (dplsDefaultDomain != null && dplsDefaultDomain.length() > 0) {
        dplsDomains.add(dplsDefaultDomain);
      }
      dplsDomains.addAll(loginService.getDomains());
    }
    Vector<String> domains = new Vector<String>(dplsDomains.size());
    domains.addAll(dplsDomains);
    return domains;
  }

  
}
