package org.clazzes.login.authorization.context;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.*;

/**
 * Object serializable to/from URL to hold the security context information.
 */
public class SecurityContextData {
    private static final Logger log = LoggerFactory.getLogger(SecurityContextData.class);

    /**
     * The hostname of the string serialized SecurityContextUrl.
     *
     * It is ignored.
     */
    public static final String SECURITY_CONTEXT_HOSTNAME = "annotated.clazzes.org";

    private final SecurityAction action;
    private final String context;

    public SecurityContextData(SecurityAction action, String context) {
        this.action = action;
        this.context = context;
    }

    public SecurityAction getAction() {
        return this.action;
    }

    public String getPath() {
        if (this.context == null) {
            return "";
        }
        return this.context;
    }

    private static final URLStreamHandler OwnHandler = new URLStreamHandler() {
        @Override
        protected URLConnection openConnection(URL url) {
            return null;
        }
    };

    /**
     * A URL formed representation of it.
     */
    public String toExternalForm() {
        try {
            // context is in Path-Format (Slashes) or empty
            String path = getPath();
            if (!path.startsWith("/")) {
                path = "/" + path; // not optimal, but should be seldom
            }
            String file = path;

            URL uu = new URL(getAction().toString(), SECURITY_CONTEXT_HOSTNAME, -1, file, OwnHandler);
            return uu.toExternalForm();
        } catch(MalformedURLException e) {
            log.error("Malformed Context", e);
            return null;
        }
    }

    /**
     * Parses ext (as written by toExternalForm) into the SecurityAction.
     * Is needed, because the clazzes.org framework is dealing with String
     * as its SecurityContext type.
     * In SVN (Commit 2947) there was code for parsing query parameter into extra information.
     */
    public static SecurityContextData fromExternalForm(String ext) {
        try {
            URL uu = new URL(null, ext, OwnHandler);
            SecurityAction securityAction = SecurityAction.valueOfIgnoreCase(uu.getProtocol());
            return new SecurityContextData(securityAction, uu.getPath());
        } catch (MalformedURLException e) {
            log.error("Malformed Context", e);
            return null;
        }
    }

    @Override
    public String toString() {
        String result = toExternalForm();
        if (result != null) {
            return result;
        }
        return "MALFORMED: ActionAndContext{" +
                "action=" + this.action +
                ", context='" + this.context + '\'' +
                '}';
    }
}
