/*
 * Decompiled with CFR 0.152.
 */
package org.clazzes.login.asn1.tools;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigInteger;
import java.util.BitSet;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.TreeMap;
import org.clazzes.login.asn1.annotations.ASN1Field;
import org.clazzes.login.asn1.annotations.ASN1Sequence;
import org.clazzes.login.asn1.annotations.ASN1Type;
import org.clazzes.login.asn1.annotations.ASN1TypeCase;
import org.clazzes.login.asn1.annotations.ASN1TypeSwitch;
import org.clazzes.login.asn1.streams.ConfinedInputStream;
import org.clazzes.login.asn1.tools.ASN1Exception;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.Oid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ASN1Helper {
    private static final Logger log = LoggerFactory.getLogger(ASN1Helper.class);

    public static void writeLength(OutputStream os, int len) throws IOException {
        if (len < 0) {
            throw new IllegalArgumentException("Negative length for ASN.1");
        }
        if (len < 128) {
            os.write(len);
        } else {
            byte[] tmp = new byte[5];
            tmp[1] = (byte)(len >>> 24);
            tmp[2] = (byte)(len >>> 16);
            tmp[3] = (byte)(len >>> 8);
            tmp[4] = (byte)len;
            if (tmp[1] != 0) {
                tmp[0] = -124;
                os.write(tmp);
            } else if (tmp[2] != 0) {
                tmp[1] = -125;
                os.write(tmp, 1, 4);
            } else if (tmp[3] != 0) {
                tmp[2] = -126;
                os.write(tmp, 2, 3);
            } else {
                tmp[3] = -127;
                os.write(tmp, 3, 2);
            }
        }
    }

    public static int readLength(InputStream is) throws IOException {
        int b1 = is.read();
        if (b1 < 0) {
            throw new EOFException("EOF in first ASN.1 length byte.");
        }
        if (b1 < 128) {
            return b1;
        }
        int n = b1 & 0x7F;
        if (n < 1 || n > 4) {
            throw new ASN1Exception("Extendeg length byte [" + b1 + "] is not in the range [0x81,0x84].");
        }
        int ret = 0;
        while (--n >= 0) {
            int b = is.read();
            if (b < 0) {
                throw new EOFException("EOF in followup ASN.1 length byte.");
            }
            ret = ret << 8 | b;
        }
        return ret;
    }

    protected static void readExpectedTag(InputStream is, ASN1Type type) throws IOException {
        int rtag = is.read();
        if (rtag != type.getTag()) {
            throw new ASN1Exception("Read tag [" + rtag + "] is not the expected tag [" + String.valueOf((Object)type) + "].");
        }
    }

    protected static byte[] readBytes(InputStream is, ASN1Type type) throws IOException {
        ASN1Helper.readExpectedTag(is, type);
        int len = ASN1Helper.readLength(is);
        byte[] ret = new byte[len];
        is.read(ret);
        return ret;
    }

    protected static void writeBytes(OutputStream os, ASN1Type type, byte[] bytes) throws IOException {
        os.write(type.getTag());
        ASN1Helper.writeLength(os, bytes.length);
        os.write(bytes);
    }

    public static byte[] readOctetString(InputStream is) throws IOException {
        return ASN1Helper.readBytes(is, ASN1Type.OCTET_STRING);
    }

    public static void writeOctetString(OutputStream os, byte[] bytes) throws IOException {
        ASN1Helper.writeBytes(os, ASN1Type.OCTET_STRING, bytes);
    }

    public static byte[] readBitStringBytes(InputStream is) throws IOException {
        ASN1Helper.readExpectedTag(is, ASN1Type.BIT_STRING);
        int len = ASN1Helper.readLength(is);
        if (len <= 0) {
            throw new ASN1Exception("Zero length in BIT STRING.");
        }
        int unused = is.read();
        if (unused != 0) {
            log.warn("Ignoring unused bit count [{}] in BIT STRING.", (Object)unused);
        }
        byte[] ret = new byte[len - 1];
        is.read(ret);
        return ret;
    }

    public static void writeBitStringBytes(OutputStream os, byte[] bytes) throws IOException {
        os.write(ASN1Type.BIT_STRING.getTag());
        ASN1Helper.writeLength(os, bytes.length + 1);
        os.write(0);
        os.write(bytes);
    }

    public static BitSet readBitString(InputStream is) throws IOException {
        ASN1Helper.readExpectedTag(is, ASN1Type.BIT_STRING);
        int len = ASN1Helper.readLength(is);
        if (len <= 0) {
            throw new ASN1Exception("Zero length in BIT STRING.");
        }
        int unused = is.read();
        if (unused < 0 || unused > 7) {
            throw new ASN1Exception("Invalid unused bits [" + unused + "] in BIT STRING.");
        }
        byte[] bytes = new byte[len - 1];
        is.read(bytes);
        int nbits = (len - 1) * 8 - unused;
        BitSet ret = new BitSet(nbits);
        for (int i = 0; i < nbits; ++i) {
            ret.set(i, (bytes[i / 8] & 1 << 7 - (i & 7)) != 0);
        }
        return ret;
    }

    public static void writeBitString(OutputStream os, BitSet bs) throws IOException {
        os.write(ASN1Type.BIT_STRING.getTag());
        byte[] bytes = new byte[(bs.length() + 7) / 8];
        for (int i = 0; i < bs.length(); ++i) {
            if (!bs.get(i)) continue;
            int n = i / 8;
            bytes[n] = (byte)(bytes[n] | 1 << 7 - (i & 7));
        }
        ASN1Helper.writeLength(os, bytes.length + 1);
        os.write(bytes.length * 8 - bs.length());
        os.write(bytes);
    }

    public static BigInteger readInteger(InputStream is) throws IOException {
        byte[] bytes = ASN1Helper.readBytes(is, ASN1Type.INTEGER);
        return new BigInteger(bytes);
    }

    public static void writeInteger(OutputStream os, BigInteger i) throws IOException {
        byte[] bytes = i.toByteArray();
        ASN1Helper.writeBytes(os, ASN1Type.INTEGER, bytes);
    }

    public static Oid readOid(InputStream is) throws IOException {
        try {
            return new Oid(is);
        }
        catch (GSSException e) {
            throw new ASN1Exception("Invalid OBJECT IDENTIFIER", e);
        }
    }

    public static void writeOid(OutputStream os, Oid oid) throws IOException {
        try {
            os.write(oid.getDER());
        }
        catch (GSSException e) {
            throw new ASN1Exception("Invalid Oid instance given for OBJECT IDENTIFIER", e);
        }
    }

    public static Void readNull(InputStream is) throws IOException {
        ASN1Helper.readExpectedTag(is, ASN1Type.NULL);
        int c = is.read();
        if (c != 0) {
            throw new ASN1Exception("Invalid content byte [" + c + "] for NULL");
        }
        return null;
    }

    public static void writeNull(OutputStream os) throws IOException {
        os.write(ASN1Type.NULL.getTag());
        os.write(0);
    }

    protected static SortedMap<Integer, Field> getASN1Fields(Class<?> cls) throws ASN1Exception {
        if (cls.getAnnotation(ASN1Sequence.class) == null) {
            throw new ASN1Exception("Object of type [" + String.valueOf(cls) + "] is no ASN.1 sequence.");
        }
        TreeMap<Integer, Field> fields = new TreeMap<Integer, Field>();
        while (cls != null && cls != Object.class) {
            for (Field f : cls.getDeclaredFields()) {
                ASN1Field asn1Field = f.getAnnotation(ASN1Field.class);
                if (asn1Field == null) continue;
                int order = asn1Field.order();
                fields.put(order, f);
            }
            cls = cls.getSuperclass();
        }
        return fields;
    }

    protected static Method getASN1TypeSwitch(Class<?> cls) throws ASN1Exception {
        if (cls.getAnnotation(ASN1Sequence.class) == null) {
            throw new ASN1Exception("Object of type [" + String.valueOf(cls) + "] is no ASN.1 sequence.");
        }
        while (cls != null && cls != Object.class) {
            for (Method m : cls.getDeclaredMethods()) {
                if (m.getAnnotation(ASN1TypeSwitch.class) == null) continue;
                return m;
            }
            cls = cls.getSuperclass();
        }
        return null;
    }

    public static Object readObject(InputStream is, ASN1Type type, Class<?> cls) throws IOException {
        switch (type) {
            case BIT_STRING: {
                if (BitSet.class == cls) {
                    return ASN1Helper.readBitString(is);
                }
                if (byte[].class == cls) {
                    return ASN1Helper.readBitStringBytes(is);
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 BIT STRING.");
            }
            case INTEGER: {
                if (BigInteger.class == cls) {
                    return ASN1Helper.readInteger(is);
                }
                if (Integer.TYPE == cls || Integer.class == cls) {
                    BigInteger bi = ASN1Helper.readInteger(is);
                    return bi.intValueExact();
                }
                if (Long.TYPE == cls || Long.class == cls) {
                    BigInteger bi = ASN1Helper.readInteger(is);
                    return bi.longValueExact();
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 INTEGER.");
            }
            case OBJECT_IDENTIFIER: {
                if (Oid.class == cls) {
                    return ASN1Helper.readOid(is);
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 OBJECT IDENTIFIER.");
            }
            case OCTET_STRING: {
                if (byte[].class == cls) {
                    return ASN1Helper.readOctetString(is);
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 OCTET STRING.");
            }
            case SEQUENCE: {
                return ASN1Helper.readComplex(is, cls);
            }
            case NULL: {
                return ASN1Helper.readNull(is);
            }
        }
        throw new ASN1Exception("Reading objects of type [" + String.valueOf((Object)type) + "] is not supported.");
    }

    public static void readField(InputStream is, Object x, Field f) throws IOException {
        ASN1Field asn1Field = f.getAnnotation(ASN1Field.class);
        if (log.isDebugEnabled()) {
            log.info("Reading field [{}]", (Object)f);
        }
        try {
            PropertyDescriptor pd = new PropertyDescriptor(f.getName(), f.getDeclaringClass());
            Object value = ASN1Helper.readObject(is, asn1Field.type(), f.getType());
            pd.getWriteMethod().invoke(x, value);
        }
        catch (IntrospectionException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
            throw new ASN1Exception("Reflection error writing field [" + String.valueOf(f) + "]", e);
        }
    }

    public static <T> T readComplex(InputStream is, Class<T> cls) throws IOException {
        Method typeSwitch = ASN1Helper.getASN1TypeSwitch(cls);
        ASN1Helper.readExpectedTag(is, ASN1Type.SEQUENCE);
        int len = ASN1Helper.readLength(is);
        try (ConfinedInputStream cis = new ConfinedInputStream(is, len);){
            T x;
            Class<Object> subCls;
            if (typeSwitch == null) {
                subCls = cls;
            } else {
                int intDesignator;
                String designator;
                ASN1TypeSwitch ts = typeSwitch.getAnnotation(ASN1TypeSwitch.class);
                if (typeSwitch.getReturnType() == Oid.class) {
                    designator = ASN1Helper.readOid(cis).toString();
                    intDesignator = Integer.MIN_VALUE;
                } else {
                    intDesignator = ASN1Helper.readInteger(cis).intValueExact();
                    designator = "";
                }
                subCls = null;
                for (ASN1TypeCase tc : ts.value()) {
                    if (!designator.equals(tc.oid()) || intDesignator != tc.value()) continue;
                    subCls = tc.clazz();
                    break;
                }
                if (subCls == null) {
                    throw new ASN1Exception("No subclass for type switch value [" + designator + "] of class [" + String.valueOf(cls) + "]");
                }
            }
            try {
                x = subCls.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
            catch (IllegalAccessException | IllegalArgumentException | InstantiationException | NoSuchMethodException | SecurityException | InvocationTargetException e) {
                throw new ASN1Exception("Reflection error instantiating [" + String.valueOf(subCls) + "]", e);
            }
            SortedMap<Integer, Field> fields = ASN1Helper.getASN1Fields(subCls);
            for (Field f : fields.values()) {
                ASN1Helper.readField(cis, x, f);
            }
            Iterator<Field> iterator = x;
            return (T)iterator;
        }
    }

    public static <T> T decodeComplex(byte[] encoded, Class<T> cls) throws IOException {
        try (ConfinedInputStream is = ConfinedInputStream.wrap(encoded);){
            T t = ASN1Helper.readComplex(is, cls);
            return t;
        }
    }

    public static void writeObject(OutputStream os, ASN1Type type, Object x) throws IOException {
        Class<?> cls = x == null ? null : x.getClass();
        switch (type) {
            case BIT_STRING: {
                if (BitSet.class == cls) {
                    ASN1Helper.writeBitString(os, (BitSet)x);
                    break;
                }
                if (byte[].class == cls) {
                    ASN1Helper.writeBitStringBytes(os, (byte[])x);
                    break;
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 BIT STRING.");
            }
            case INTEGER: {
                if (BigInteger.class == cls) {
                    ASN1Helper.writeInteger(os, (BigInteger)x);
                    break;
                }
                if (Integer.TYPE == cls || Integer.class == cls) {
                    ASN1Helper.writeInteger(os, BigInteger.valueOf(((Integer)x).intValue()));
                    break;
                }
                if (Long.TYPE == cls || Long.class == cls) {
                    ASN1Helper.writeInteger(os, BigInteger.valueOf((Long)x));
                    break;
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 INTEGER.");
            }
            case OBJECT_IDENTIFIER: {
                if (Oid.class == cls) {
                    ASN1Helper.writeOid(os, (Oid)x);
                    break;
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 OBJECT IDENTIFIER.");
            }
            case OCTET_STRING: {
                if (byte[].class == cls) {
                    ASN1Helper.writeOctetString(os, (byte[])x);
                    break;
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 OCTET STRING.");
            }
            case SEQUENCE: {
                ASN1Helper.writeComplex(os, x);
                break;
            }
            case NULL: {
                if (x == null) {
                    ASN1Helper.writeNull(os);
                    break;
                }
                throw new IllegalArgumentException("Unsupported type [" + String.valueOf(cls) + "] for ASN.1 NULL.");
            }
            default: {
                throw new ASN1Exception("Writing objects of type [" + String.valueOf((Object)type) + "] is not supported.");
            }
        }
    }

    public static void writeField(OutputStream os, Object x, Field f) throws IOException {
        if (log.isDebugEnabled()) {
            log.info("Writing field [{}]", (Object)f);
        }
        ASN1Field asn1Field = f.getAnnotation(ASN1Field.class);
        try {
            PropertyDescriptor pd = new PropertyDescriptor(f.getName(), f.getDeclaringClass());
            Object v = pd.getReadMethod().invoke(x, new Object[0]);
            ASN1Helper.writeObject(os, asn1Field.type(), v);
        }
        catch (IntrospectionException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
            throw new ASN1Exception("Reflection error reading field [" + String.valueOf(f) + "]", e);
        }
    }

    public static void writeComplex(OutputStream os, Object x) throws IOException {
        SortedMap<Integer, Field> fields = ASN1Helper.getASN1Fields(x.getClass());
        Method typeSwitch = ASN1Helper.getASN1TypeSwitch(x.getClass());
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream();){
            if (typeSwitch != null) {
                try {
                    Object designator = typeSwitch.invoke(x, new Object[0]);
                    if (typeSwitch.getReturnType() == Oid.class) {
                        ASN1Helper.writeOid(bos, (Oid)designator);
                    } else {
                        ASN1Helper.writeInteger(bos, BigInteger.valueOf(((Integer)typeSwitch.invoke(x, new Object[0])).intValue()));
                    }
                }
                catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
                    throw new ASN1Exception("Reflection error invoking type switch [" + String.valueOf(typeSwitch) + "]", e);
                }
            }
            for (Field f : fields.values()) {
                ASN1Helper.writeField(bos, x, f);
            }
            byte[] bytes = bos.toByteArray();
            ASN1Helper.writeBytes(os, ASN1Type.SEQUENCE, bytes);
        }
    }

    public static byte[] encodeComplex(Object x) throws IOException {
        try (ByteArrayOutputStream bos = new ByteArrayOutputStream(1024);){
            ASN1Helper.writeComplex(bos, x);
            byte[] byArray = bos.toByteArray();
            return byArray;
        }
    }
}

