/***********************************************************
 *
 * ASN.1 Tools of the clazzes.org project
 * https://www.clazzes.org
 *
 * Created: 24.07.2024
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

 package org.clazzes.login.asn1.streams;

import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

/**
 * A confined input stream, which represents a subset of an input stream.
 *
 * The close method throws an Exception, if not all the content is read.
 */
public class ConfinedInputStream extends NoMarkInputStream{

    private final long size;
    private long consumed;

    public ConfinedInputStream(InputStream in, long size) {
        super(in);
        this.size = size;
        this.consumed = 0L;
    }

    /**
     * Wrap the given byte array as a confined input stream.
     * @param bytes The byte array to parse
     * @return A confied input stream, which throws an expcetion in {@link #close()},
     *         if not all bytes are consumed.
     */
    public static ConfinedInputStream wrap(byte[] bytes) {
        return new ConfinedInputStream(new ByteArrayInputStream(bytes),bytes.length);
    }

    protected int getRemaining() {
        if (this.size < 0L) {
            return Integer.MAX_VALUE;
        }

        long ret = this.size - this.consumed;

        if (ret > Integer.MAX_VALUE) {
            return Integer.MAX_VALUE;
        }

        return (int)ret;
    }

    @Override
    public int read() throws IOException {

        if (this.size >= 0L && this.consumed >= this.size) {
            return -1;
        }

        int ret = super.read();

        if (ret >= 0) {
            ++this.consumed;
        }
        else {
            throw new EOFException("Premature EOF in after ["+
                                   this.consumed+
                                   "] bytes consumed.");
        }

        return ret;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {

        int rem = this.getRemaining();

        if (rem == 0) {
            return -1;
        }

        int ret = super.read(b,off,Math.min(len,rem));

        if (len > 0 && ret <= 0) {
            throw new EOFException("Premature EOF after ["+
                                   this.consumed+"/"+this.size+
                                   "] bytes consumed.");
        }

        if (ret > 0) {
            this.consumed += ret;
        }

        return ret;
    }

    @Override
    public int available() throws IOException {

        return Math.min(super.available(),this.getRemaining());
    }

    @Override
    public void close() throws IOException {

        if (this.size >= 0L && this.consumed < this.size) {
            throw new EOFException("Partial read with only ["+
                                   this.consumed+"/"+this.size+
                                   "] bytes consumed.");
        }
    }

    public long getConsumed() {
        return this.consumed;
    }

}
