/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.ha;

import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.clazzes.util.aop.ThreadLocalManager;

/**
 * An interceptor locking on a given IHaLockManger instance.
 */
public class HaLockInterceptor implements MethodInterceptor {

    private IHaLockManager haLockManager;
    private String threadLocalKey;
    
    /* (non-Javadoc)
     * @see org.aopalliance.intercept.MethodInterceptor#invoke(org.aopalliance.intercept.MethodInvocation)
     */
    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
        
        boolean bound = false;
        
        try (IHaLock lock = this.haLockManager.doLock()) {
         
            if (lock != null && this.threadLocalKey != null) {
                ThreadLocalManager.bindResource(this.threadLocalKey,lock);
                bound = true;
            }
            
            return invocation.proceed();
        }
        finally {
            if (bound) {
                ThreadLocalManager.unbindResource(this.threadLocalKey);
            }
        }
    }

    /**
     * @return The configure lock manager.
     */
    public IHaLockManager getHaLockManager() {
        return this.haLockManager;
    }

    /**
     * @param haLockManager The lock manager to use for locking.
     */
    public void setHaLockManager(IHaLockManager haLockManager) {
        this.haLockManager = haLockManager;
    }

    /**
     * @return The configured  key under which the lock will be bound by 
     *     {@link ThreadLocalManager#bindResource(String, Object)}
     */
    public String getThreadLocalKey() {
        return this.threadLocalKey;
    }

    /**
     * @param threadLocalKey The key under which the lock will be bound
     *     to the current thread by
     *     {@link ThreadLocalManager#bindResource(String, Object)}
     *     to configure.
     */
    public void setThreadLocalKey(String threadLocalKey) {
        this.threadLocalKey = threadLocalKey;
    }

}
