/***********************************************************
 * $Id: HiLoIdGenerator.java 794 2013-02-19 15:46:41Z util $
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql.dao;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.function.Consumer;

import javax.sql.DataSource;

import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.sql.SQLDialectFactory;
import org.clazzes.util.sql.helper.JDBCHelper;
import org.clazzes.util.sql.helper.JDBCTransaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>A Hi-Lo ID generator, which generates IDs in block with
 * a configurable bock size.</p>
 *
 * <p>Be sure to call {@link #initialize()} after setting a valid datasource
 * via {@link #setDataSource(DataSource)} and actually creating your generator
 * database table.</p>
 *
 * <p>The code is placed in this abstract class, since some details have to be
 * done in slightly different way when dealing with a Hibernate-controlled database.
 * This means, usually the subclass HiLoIdGenerator should be used.  Only when
 * supporting a database Hibernate controls, the other subclass HibernateHiLoIdGenerator
 * should be used, as hibernate stores values divided by the block size in the
 * id table, whereas HiLoIdGenerator stores values without division. </p>
 */
public abstract class AbstractHiLoIdGenerator implements IdGenerator {

    private static final Logger log = LoggerFactory.getLogger(HiLoIdGenerator.class);

    public static final String DEFAULT_ID_TABLE_NAME = "ID_GENERATOR";
    public static final String DEFAULT_NEXT_ID_COL_NAME = "NEXT_ID";

    private String idTableName;
    private String nextIdColumnName;
    private String loIdColumnName;
    private long blockSize;
    private DataSource dataSource;
    private String threadLocalKey;

    // Transactional state for retrying.
    private final class State {
        private Long hiValue;
        private long loValue;
        public State(Long hiValue, long loValue) {
            this.hiValue = hiValue;
            this.loValue = loValue;
        }
        public State copy() {
            return new State(this.hiValue, this.loValue);
        }

    }
    private State state = new State(null, 0);

    /** Returns the very first hi value the generator writes into the database (if none was written before).
     * @return the very first hi value the generator writes into the database (if none was written before).
     */
    protected abstract Long getInitialHiValue();

    /** Returns the step size, by which the hi value (as stored in the database) is incremented when a new
     *  block has to be started.
     * @return step size as described
     */
    protected abstract Long getHiStepSize();

    /** Returns the current base id, to which the lo value is added.  It depends on the hi value stored in the
     *  database, but there might be some factor in between that is encapsulated in this function.
     * @return the current base id as described
     */
    protected abstract Long getBaseId();

    @FunctionalInterface
    private interface TransactionFn {
        public void accept(State state, Statement statement) throws SQLException;
    }

    private boolean isRetryable(Throwable e) {
        if (e == null) {
            return false;
        }

        if (e instanceof SQLException sqlE) {
            // The 40001 sqlstate is defined by the sql standard and is mostly portable between databases.
            // Oracle uses a nonstandard sqlstate for serialization failures.
            if ("40001".equals(sqlE.getSQLState()) || sqlE.getErrorCode() == 8177) {
                return true;
            }
        }

        return isRetryable(e.getCause());
    }

    public boolean supportSelectForUpdate(Connection conn) throws SQLException {
        switch (SQLDialectFactory.getSQLDialect(conn)) {
        case DERBY:
        case POSTGRES:
        case MYSQL:
        case ORACLE:
            return true;
        case ISO:
        // Microsoft sql server has deadlocking issues with for update and has deprecated it.
        case MSSQL:
        default:
            return false;
        }
    }

    // Retryable!
    private void withTransaction(TransactionFn fn) throws SQLException {

        var existingConnection = this.threadLocalKey != null
            ? (Connection) ThreadLocalManager.getBoundResource(this.threadLocalKey)
            : null;

        // Using saveopints avoids deadlock issues if all pool connections are occupied with id generating transaction.
        boolean useSavepoint = existingConnection != null
                               && (existingConnection.getTransactionIsolation() == Connection.TRANSACTION_SERIALIZABLE
                                   || supportSelectForUpdate(existingConnection));


        if (useSavepoint) {
            var savepoint = existingConnection.setSavepoint();
            boolean successful = false;
            var newState = this.state.copy();
            try {
                try (var statement = existingConnection.createStatement()) {
                    lockRows(statement);
                    fn.accept(newState, statement);
                }
                successful = true;
            } finally {
                if (successful) {
                    try {
                        existingConnection.releaseSavepoint(savepoint);
                    } catch (SQLFeatureNotSupportedException e) {
                        // Oracle supports the creation and rollback of savepoints,
                        // but bizarely throws an error if you try to destroy them.
                    }
                } else {
                    existingConnection.rollback(savepoint);
                }
            }

            // Commit new state.
            this.state = newState;
        } else {
            while (true) {
                var newState = this.state.copy();
                try {
                    try (
                        JDBCTransaction txn = this.openSerializableTransaction();
                        var statement = txn.getConnection().createStatement();
                    ) {
                        fn.accept(newState, statement);
                        txn.commit();
                    }
                } catch (Exception e) {
                    if (isRetryable(e)) {
                        // Retry transaction on serialziation failures
                        continue;
                    }
                    throw e;
                }
                // Commit new state.
                this.state = newState;
                break;
            }
        }

    }

    public AbstractHiLoIdGenerator() {
        this.idTableName = DEFAULT_ID_TABLE_NAME;
        this.nextIdColumnName = DEFAULT_NEXT_ID_COL_NAME;
        this.blockSize = 100L;
    }

    protected JDBCTransaction openSerializableTransaction() throws SQLException {
        return new JDBCTransaction(this.dataSource,Connection.TRANSACTION_SERIALIZABLE);
    }

    private void lockRows(Statement statement) throws SQLException {
        if (statement.getConnection().getTransactionIsolation() == Connection.TRANSACTION_SERIALIZABLE) {
            // If we are in a serializable transaction the database will handle the locking for us.
            return;
        }

        // The useSavepoint logic in withTransaction should have prevented this situation.
        assert supportSelectForUpdate(statement.getConnection());

        statement.executeQuery("select * from " + this.idTableName + " for update");
    }

    protected void fetchNextId(State state, Statement statement) throws SQLException {

        ResultSet rs = statement.executeQuery("select "+
                this.nextIdColumnName
                +" from "+ this.idTableName);

        Long nextId = null;

        try {
            while (rs.next()) {

                if (nextId == null) {
                    nextId = JDBCHelper.getLong(rs,1);
                }
                else {
                    throw new DAOException("Id table ["+this.idTableName+"] contains multiple rows.");
                }
            }
        }
        finally {
            rs.close();
        }

        state.hiValue = nextId;
    }

    /**
     * Fetch hi and lo values in the case, that a lo value column has been configured.
     *
     * @param statement An SQL statement.
     * @throws SQLException Upon DB errors.
     */
    protected void fetchInitialHiLoId(State state, Statement statement) throws SQLException {

        ResultSet rs = statement.executeQuery("select "+
                this.nextIdColumnName+","+
                this.loIdColumnName+
                " from "+ this.idTableName);

        Long nextId = null;

        try {
            while (rs.next()) {

                if (nextId == null) {
                    nextId = JDBCHelper.getLong(rs,1);
                    state.loValue = rs.getLong(2);
                }
                else {
                    throw new DAOException("Id table ["+this.idTableName+"] contains multiple rows.");
                }
            }
        }
        finally {
            rs.close();
        }


        if (state.loValue <= -this.blockSize || state.loValue > 0) {

            log.warn("Table [{}] contains invalid lo value [{}], ignoring this value and starting a new block.",
                    this.idTableName,state.loValue);

            state.loValue = 0;
        }
        else if (state.loValue != 0) {

            log.warn("Table [{}] contains valid lo value [{}], resetting table lo value to zero.",
                    this.idTableName,state.loValue);

            statement.executeUpdate("update "+ this.idTableName+
                    " set "+this.loIdColumnName+" = 0");
        }

        state.hiValue = nextId;
    }


    protected void incrementHiValue(Statement statement) throws SQLException {

        StringBuffer sb = new StringBuffer();

        sb.append("update ");
        sb.append(this.idTableName);
        sb.append(" set ");
        sb.append(this.nextIdColumnName);
        sb.append("=");
        sb.append(this.nextIdColumnName);
        sb.append("+");
        sb.append(this.getHiStepSize());

        if (this.loIdColumnName != null) {
            sb.append(", ");
            sb.append(this.loIdColumnName);
            sb.append("=0");
        }

        statement.executeUpdate(sb.toString());
    }

    /**
     * An initialization function, which check the database table for a single row
     * with the next high value.
     * @throws SQLException
     */
    public void initialize() throws SQLException {

        this.withTransaction((state, statement) -> {
            if (this.loIdColumnName == null) {
                this.fetchNextId(state, statement);
                state.loValue = 0;
            }
            else {
                this.fetchInitialHiLoId(state, statement);
            }

            if (state.hiValue == null) {

                state.hiValue = this.getInitialHiValue();
                state.loValue = this.loIdColumnName == null ? 0 : 1-this.blockSize;

                log.info("Id table [{}] contains no rows, inserting initial value [{}].",
                        this.idTableName,state.hiValue);

                StringBuffer sb = new StringBuffer();

                sb.append("insert into ");
                sb.append(this.idTableName);
                sb.append(" (");
                sb.append(this.nextIdColumnName);

                if (this.loIdColumnName != null) {
                    sb.append(",");
                    sb.append(this.loIdColumnName);
                }

                sb.append(") values (");
                sb.append(state.hiValue);

                if (this.loIdColumnName != null) {
                    sb.append(",");
                    sb.append(state.loValue);
                }

                sb.append(")");

                statement.executeUpdate(sb.toString());
            }
        });

        if (log.isDebugEnabled()) {
            log.debug("Last known high value was [{}].",this.state.hiValue);
        }
    }

    public void destroy() throws SQLException {

        if (this.loIdColumnName == null) {
            log.info("No low table column set, HiLo ID Generator destroy() method will do nothing.");
            return;
        }

        if (this.state.hiValue == null || this.state.loValue >= 0) {

            log.info("ID Generator did not generate an incomplete hi block, stopping ID generator now.");
        }
        else {

            this.withTransaction((state, statement) -> {
                int n = statement.executeUpdate("update "+ this.idTableName+
                        " set "+ this.loIdColumnName+"="+ state.loValue +
                        " where "+this.nextIdColumnName+"=" + state.hiValue);

                if (n == 0) {
                    log.info("ID Generator generated an incomplete hi block for [hi={},lo={}], but another instance has taken yet-another hi block.",
                            state.hiValue,state.loValue);
                }
                else {
                    log.info("ID Generator generated an incomplete hi block for [hi={},lo={}], persisting lo value for future use.",
                            state.hiValue,state.loValue);
                }
            });
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.dao.IdGenerator#generateNext()
     */
    @Override
    public synchronized Long generateNext() {

        if (this.state.hiValue == null || this.state.loValue >= 0) {

            if (log.isDebugEnabled()) {
                log.debug("Generating new high value as successor to [{}].",this.state.hiValue);
            }

            try {
                this.withTransaction((state, statement) -> {
                    this.incrementHiValue(statement);
                    this.fetchNextId(state, statement);

                    if (state.hiValue == null) {
                        throw new DAOException("Id table [" + this.idTableName+"] contains no rows, did you call initialize()?");
                    }

                    if (log.isDebugEnabled()) {
                        log.debug("New high value is [{}].",state.hiValue);
                    }

                    state.loValue = -this.blockSize;
                });
            } catch (SQLException e) {
                throw new DAOException("Error generating new High value",e);
            }
        }

        Long ret = Long.valueOf(this.getBaseId().longValue() + this.state.loValue++);

        if (log.isDebugEnabled()) {
            log.debug("Generated new ID [{}].",ret);
        }

        return ret;
    }

    /**
     *  Getter to the hi value as stored in the corresponding database table, in order to make
     *  it available for test cases.
     * @return the hi value this generator currently operates on.
     */
    public Long getHiValue() {
        return this.state.hiValue;
    }

    /**
     * @return The name of the ID table.
     */
    public String getIdTableName() {
        return this.idTableName;
    }

    /**
     * @param idTableName THe name of the ID table. The default value
     *         is <code>"ID_GENERATOR"</code> resp. @see DEFAULT_ID_TABLE_NAME.
     */
    public void setIdTableName(String idTableName) {
        this.idTableName = idTableName;
    }

    /**
     * @return The name of the column, where the next hi value is stored.
     */
    public String getNextIdColumnName() {
        return this.nextIdColumnName;
    }

    /**
     * @param nextIdColumnName The name of the column,
     *        where the next hi value is stored. The default value is
     *         <code>"NEXT_ID"</code> resp. @see DEFAULT_NEXT_ID_COL_NAME
     */
    public void setNextIdColumnName(String nextIdColumnName) {
        this.nextIdColumnName = nextIdColumnName;
    }

    /**
     * @return The low column name used to persist the current
     *          low value for future use inside {@link #destroy()}.
     *          If <code>null</code>, the low column will not be used and each time,
     *          a new block is started, the next ID block will be used when the next
     *          ID generator instance is set up after restart of your Java process.
     */
    public String getLoIdColumnName() {
        return this.loIdColumnName;
    }

    /**
     * @param loIdColumnName The low column name used to persist the current
     *             low value for future use inside {@link #destroy()}.
     */
    public void setLoIdColumnName(String loIdColumnName) {
        this.loIdColumnName = loIdColumnName;
    }

    /**
     * @return The block size for high values.
     */
    public long getBlockSize() {
        return this.blockSize;
    }

    /**
     * @param blockSize The block size to set. The default value is
     *         <code>100</code>.
     */
    public void setBlockSize(long blockSize) {
        this.blockSize = blockSize;
    }

    /**
     * @return The datasource used to pull new connections.
     */
    public DataSource getDataSource() {
        return this.dataSource;
    }

    /**
     * @param dataSource The datasource use to pull new connections to set.
     */
    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public String getThreadLocalKey() {
        return this.threadLocalKey;
    }

    // Setting the threadLocalKey allows us to use more deadlock resistant savepoints.
    public void setThreadLocalKey(String threadLocalKey) {
        this.threadLocalKey = threadLocalKey;
    }
}
