/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.criteria;

import org.clazzes.util.sql.SQLDialect;
import org.clazzes.util.sql.SQLFragment;

/**
 * A convenience class to aid the creation of complex sql statements by wrapping {@link SQLFragment}s
 * 
 * @author jpayr
 *
 */
public class SQLConstruct implements SQLFragment {
	
	private static final long serialVersionUID = -5890504692911862897L;

	protected enum ConstructType {
		PARENTHESIS,
		NONE
	}
	
	protected enum LinkType {
		COMMA,
		NONE
	}
	
	private SQLFragment[] content;
	
	private ConstructType type;
	
	private LinkType link;
	
	protected SQLConstruct(ConstructType type, LinkType link, SQLFragment... contents) {
		this.type = type;
		this.link = link;
		this.content = contents;
	}
	
	public static SQLConstruct parenthesis(SQLFragment... contents) {
		return new SQLConstruct(ConstructType.PARENTHESIS, LinkType.NONE, contents);
	}
	
	public static SQLConstruct commaParenthesis(SQLFragment... contents) {
		return new SQLConstruct(ConstructType.PARENTHESIS, LinkType.COMMA, contents);
	}
	
	public static SQLConstruct commaSeperated(SQLFragment... contents) {
		return new SQLConstruct(ConstructType.NONE, LinkType.COMMA, contents);
	}

	/* (non-Javadoc)
	 * @see org.clazzes.topicmaps.jdbc.sql.SQLFragment#toSQL(org.clazzes.topicmaps.jdbc.sql.SQLDialect)
	 */
	@Override
	public String toSQL(SQLDialect dialect) {
		StringBuffer result = new StringBuffer();
		
		switch (this.type) {
		case PARENTHESIS:
			switch (dialect) {
			default:
				result.append("(");
				list(result, dialect);
				result.append(")");
				break;
			}
			break;
		
		case NONE:
			switch (dialect) {
			default:
				list(result, dialect);
				break;
			}
			
		default:
			break;
		}
		
		return result.toString();
	}
	
	private void list(StringBuffer result, SQLDialect dialect) {
		for (int i=0; i < this.content.length; i++) {
			result.append(this.content[i].toSQL(dialect));
			
			if (this.content.length - i > 1) {
				switch (this.link) {
				case COMMA:
					result.append(", ");
					break;
					
				case NONE:
					result.append(" ");
					break;
					
				default:
					break;
				}
			}
		}
	}

}
