/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.criteria;

import java.util.Arrays;

import org.clazzes.util.sql.SQLDialect;
import org.clazzes.util.sql.SQLFragment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A convenience class for wrapping condition testing for sql statement generation
 * 
 * @author jpayr
 *
 */
public class SQLCondition implements SQLFragment {
	
    private static final Logger log = LoggerFactory
            .getLogger(SQLCondition.class);
    
	/**
	 * An internal Enumeration to denote the types of comparison a condition
 	 * can represent
	 * 
	 * @author jpayr
	 */
	private enum OperationType {
		AND,
		OR,
		LIKE,
		NOT_LIKE,
		IN,
		EQ,
		NOT_EQ,
		GT_EQ,
		GT,
		LT_EQ,
		LT,
		IS_NULL,
		NOT_NULL,
		NOOP,
        REGEXP_LIKE,
        NOT_REGEXP_LIKE
	}
	
	private static final long serialVersionUID = 7213686119989723324L;

	private final SQLFragment left;
	private final SQLFragment right;
	
	private final OperationType opType;
	
	protected SQLCondition(OperationType op, SQLFragment left, SQLFragment right) {
		super();
		this.left = left;
		this.right = right;
		this.opType = op;
	}
	
	protected static SQLCondition and(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.AND, operand1, operand2);
	}
	
	public static SQLCondition and(SQLFragment...operands) {
		if (operands.length == 0)
			return null;
		else if (operands.length == 1)
			return new SQLCondition(OperationType.NOOP, operands[0], null);
		else if (operands.length == 2)
			return and(operands[0], operands[1]);
		else
			return and(operands[0], and(Arrays.copyOfRange(operands, 1, operands.length)));
	}
	
	protected static SQLCondition or(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.OR, operand1, operand2);
	}
	
	public static SQLCondition or(SQLFragment... operands) {
		if (operands.length == 0)
			return null;
		else if (operands.length == 1)
			return new SQLCondition(OperationType.NOOP, operands[0], null);
		else if (operands.length == 2) 
			return or(operands[0], operands[1]);
		else
			return or(operands[0], or(Arrays.copyOfRange(operands, 1, operands.length)));
	}
	
	public static SQLCondition eq(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.EQ, operand1, operand2);
	}
	
	public static SQLCondition notEq(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.NOT_EQ, operand1, operand2);
	}
	
	public static SQLCondition gt(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.GT, operand1, operand2);
	}
	
	public static SQLCondition gtEquals(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.GT_EQ, operand1, operand2);
	}
	
	public static SQLCondition lt(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.LT, operand1, operand2);
	}

	public static SQLCondition ltEquals(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.LT_EQ, operand1, operand2);
	}

	public static SQLCondition isNull(SQLFragment operand1) {
		return new SQLCondition(OperationType.IS_NULL, operand1, null);
	}

	public static SQLCondition notNull(SQLFragment operand1) {
		return new SQLCondition(OperationType.NOT_NULL, operand1, null);
	}
	
	public static SQLCondition like(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.LIKE, operand1, operand2);
	}
	
	public static SQLCondition in(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.IN, operand1, operand2);
	}
	
	public static SQLCondition notLike(SQLFragment operand1, SQLFragment operand2) {
		return new SQLCondition(OperationType.NOT_LIKE, operand1, operand2);
	}

    /**
     * <p>Construct a regexp comparison condition.</p>
     * 
     * <p>Please note, that Microsoft SQL Server and Derby do not have native support
     * for regular expressions, so for these dialects this function falls back
     * to {@link #like(SQLFragment, SQLFragment)}.</p>
     * 
     * @param operand The string or placeholder to match.
     * @param regex The regular expression to match with.
     * @return The regexp condition.
     */
    public static SQLCondition regexpLike(SQLFragment operand, SQLFragment regex) {
        return new SQLCondition(OperationType.REGEXP_LIKE, operand, regex);
    }

    /**
     * <p>Construct a negative regexp comparison condition.</p>
     * 
     * <p>Please note, that Microsoft SQL Server and Derby do not have native support
     * for regular expressions, so for these dialects this function falls back
     * to {@link #notLike(SQLFragment, SQLFragment)}.</p>
     * 
     * @param operand The string or placeholder to not match.
     * @param regex The regular expression to exclude.
     * @return The excluding regexp condition.
     */
    public static SQLCondition notRegexpLike(SQLFragment operand, SQLFragment regex) {
        return new SQLCondition(OperationType.NOT_REGEXP_LIKE, operand, regex);
    }

    /* (non-Javadoc)
	 * @see org.clazzes.topicmaps.jdbc.sql.criteria.SQLCriteria#toSQL(org.clazzes.topicmaps.jdbc.sql.SQLDialect)
	 */
	@Override
	public String toSQL(SQLDialect dialect) {
		StringBuffer result = new StringBuffer();
		switch (this.opType) {
		case GT:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(">");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;

		case GT_EQ:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(">=");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
			
		case LT:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append("<");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case LT_EQ:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append("<=");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case EQ:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append("=");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
			
		case NOT_EQ:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append("<>");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case IS_NULL:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" IS NULL");
				break;
			}
			break;
		
		case NOT_NULL:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" IS NOT NULL");
				break;
			}
			break;
		
		case AND:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" AND ");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case OR:
			switch (dialect) {
			default:
				result.append("(");
				result.append(this.left.toSQL(dialect));
				result.append(" OR ");
				result.append(this.right.toSQL(dialect));
				result.append(")");
				break;
			}
			break;
			
		case LIKE:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" LIKE ");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case NOT_LIKE:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" NOT LIKE ");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case IN:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				result.append(" IN ");
				result.append(this.right.toSQL(dialect));
				break;
			}
			break;
		
		case NOOP:
			switch (dialect) {
			default:
				result.append(this.left.toSQL(dialect));
				break;
			}
			break;
			
        case REGEXP_LIKE:
            switch (dialect) {
            case ORACLE:
                result.append("REGEXP_LIKE( ");
                result.append(this.left.toSQL(dialect));
                result.append(", ");
                result.append(this.right.toSQL(dialect));
                result.append(" )");
                break;
            case MYSQL:
                result.append(this.left.toSQL(dialect));
                result.append(" REGEXP ");
                result.append(this.right.toSQL(dialect));
                break;
            case POSTGRES:
                result.append(this.left.toSQL(dialect));
                result.append(" ~ ");
                result.append(this.right.toSQL(dialect));
                break;
            default:
                log.warn("REGEXP_LIKE not support with dialect [{}], using LIKE instead.",dialect);
                result.append(this.left.toSQL(dialect));
                result.append(" LIKE ");
                result.append(this.right.toSQL(dialect));
                break;
            }
            break;

        case NOT_REGEXP_LIKE:
            switch (dialect) {
            case ORACLE:
                result.append("NOT REGEXP_LIKE( ");
                result.append(this.left.toSQL(dialect));
                result.append(", ");
                result.append(this.right.toSQL(dialect));
                result.append(" )");
                break;
            case MYSQL:
                result.append(this.left.toSQL(dialect));
                result.append(" NOT REGEXP ");
                result.append(this.right.toSQL(dialect));
                break;
            case POSTGRES:
                result.append(this.left.toSQL(dialect));
                result.append(" !~ ");
                result.append(this.right.toSQL(dialect));
                break;
            default:
                log.warn("NOT_REGEXP_LIKE not support with dialect [{}], using LIKE instead.",dialect);
                result.append(this.left.toSQL(dialect));
                result.append(" NOT LIKE ");
                result.append(this.right.toSQL(dialect));
                break;
            }
            break;
		}
		
		return result.toString();
	}

	
	
}
