/***********************************************************
 * $Id$
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql.dao;

import java.util.Collection;
import java.util.List;

/**
 * A basic DAO for entities, which do not have a simple primary key.
 *
 * @param <DTO> The DTO class.
 */
public interface IBasicDAO<DTO> {

	/**
	 * Save the given DTO an generate a new ID.
	 * In the case of UUID generation for String IDs, this method
	 * generates a random UUID, which is checked against the database
	 * for duplicates.
	 *
	 * @param dto The DTO to insert.
	 * @return The passed DTO instance with a new generated ID.
	 */
	public DTO save(DTO dto);

	/**
	 * Save the given list of DTOs in a database batch and generate
	 * new IDs for them.
	 * In the case of UUID generation for String IDs, this method
	 * generates a random UUID, which is not checked against the database
	 * for duplicates for speed reasons.
	 *
	 * @param dtos The list of DTOs to insert.
	 * @return The passed DTO list with new generated IDs set.
	 */
	public List<DTO> saveBatch(List<DTO> dtos);

	/**
	 * @param dto The DTO to update
	 * @return The update count, which is zero, if the
	 *         given DTO had an invalid ID, which could not be found in the DB.
	 */
	public int update(DTO dto);

	/**
	 * Update a collection of objects as an SQL batch. This is an optional method,
	 * which throws an {@link UnsupportedOperationException} in the default
	 * implementation.
	 *
	 * @param dtos A collection of DTOs to update
	 * @return An array of <code>dtos.size()</code> update counts per
	 *         dto to be updated.
	 */
	public int[] updateBatch(Collection<DTO> dtos);

	/**
	 * @return Get all DAO objects from a database table.
	 */
	public List<DTO> getAll();

	/**
	 * @return Get all DAO objects from a database table.
	 */
	public default List<DTO> getAllForUpdate() {
		throw new UnsupportedOperationException();
	}
}