/***********************************************************
 * $Id$
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql;

import java.sql.PreparedStatement;

import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLConstruct;
import org.clazzes.util.sql.criteria.SQLOrder;
import org.clazzes.util.sql.criteria.SQLValue;

/**
 * A generator for simple SQL-statements, providing support for different SQL-dialects.
 * <br><br>
 * This is the core class to the sql-package. It provides methods to generate the most frequently used SQL statements
 * which can be inserted directly to create {@link PreparedStatement}s.
 *
 * @author jpayr
 *
 */
public class SQLStatementGenerator {

	private SQLDialect dialect;

	/**
	 * @param dialect the {@link SQLDialect} to use
	 */
	public SQLStatementGenerator(SQLDialect dialect) {
		super();
		this.dialect = dialect;
	}

	/**
	 * Generates a <code>SELECT</code> statement for the given {@link SQLDialect} as follows:
	 * <br><br>
	 * For <code>columnNames</code> == <code>null</code>:<br><br>
	 * <code>SELECT * FROM tableName [WHERE condition]</code><br>
	 * <br>
	 * For <code>columnNames</code> != <code>null</code>:<br><br>
	 * <code>SELECT tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
	 *
	 * @param tableName the {@link String} name of the table to be queried
	 * @param columnNames the {@link String} names of the columns to fetch
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 *
	 * @return a {@link String} <code>SELECT</code> statement
	 */
	public String select(String tableName, SQLValue[] columnNames, SQLCondition condition) {
	  return selectInternal(tableName, columnNames, condition,null,false);
  }

  /**
   * Generates a <code>SELECT</code> statement for the given {@link SQLDialect} as follows:
   * <br><br>
   * For <code>columnNames</code> == <code>null</code>:<br><br>
   * <code>SELECT * FROM tableName [WHERE condition]</code><br>
   * <br>
   * For <code>columnNames</code> != <code>null</code>:<br><br>
   * <code>SELECT tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
   *
   * @param tableName the {@link String} name of the table to be queried
   * @param columnNames the {@link String} names of the columns to fetch
   * @param condition the criteria by which to select or <code>null</code> if no criteria exists
   * @param orders an optional array of {@link SQLOrder} objects
   *
   * @return a {@link String} <code>SELECT</code> statement
   */
    public String select(String tableName, SQLValue[] columnNames, SQLCondition condition, SQLOrder[] orders) {
		return selectInternal(tableName,columnNames,condition,orders,false);
    }

	/**
	 * Generates a <code>SELECT FOR UPDATE</code> statement for
	 * the given {@link SQLDialect} as follows:
	 * <br><br>
	 * For <code>columnNames</code> == <code>null</code>:<br><br>
	 * <code>SELECT FOR UPDATE * FROM tableName [WHERE condition]</code><br>
	 * <br>
	 * For <code>columnNames</code> != <code>null</code>:<br><br>
	 * <code>SELECT FOR UPDATE tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
	 *
	 * @param tableName the {@link String} name of the table to be queried
	 * @param columnNames the {@link String} names of the columns to fetch
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 *
	 * @return a {@link String} <code>SELECT</code> statement
	 */
	public String selectForUpdate(String tableName, SQLValue[] columnNames, SQLCondition condition) {
		return selectInternal(tableName, columnNames, condition,null,true);
	}

	/**
	 * Generates a <code>SELECT FOR UPDATE</code> statement for
	 * the given {@link SQLDialect} as follows:
	 * <br><br>
	 * For <code>columnNames</code> == <code>null</code>:<br><br>
	 * <code>SELECT FOR UPDATE * FROM tableName [WHERE condition]</code><br>
	 * <br>
	 * For <code>columnNames</code> != <code>null</code>:<br><br>
	 * <code>SELECT FOR UPDATE tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
	 *
	 * @param tableName the {@link String} name of the table to be queried
	 * @param columnNames the {@link String} names of the columns to fetch
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 * @param orders an optional array of {@link SQLOrder} objects
	 *
	 * @return a {@link String} <code>SELECT</code> statement
	 */
	  public String selectForUpdate(String tableName, SQLValue[] columnNames, SQLCondition condition, SQLOrder[] orders) {
		  return selectInternal(tableName,columnNames,condition,orders,true);
	  }

	/**
	 * Generates a <code>SELECT</code> statement for the given {@link SQLDialect} as follows:
	 * <br><br>
	 * For <code>columnNames</code> == <code>null</code>:<br><br>
	 * <code>SELECT * FROM tableName [WHERE condition]</code><br>
	 * <br>
	 * For <code>columnNames</code> != <code>null</code>:<br><br>
	 * <code>SELECT tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
	 *
	 * @param tableName the {@link String} name of the table to be queried
	 * @param columnNames the {@link String} names of the columns to fetch
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 * @param orders an optional array of {@link SQLOrder} objects
	 * @param forUpdate If this statement
	 *
	 * @return a {@link String} <code>SELECT</code> statement
	 */
	protected String selectInternal(String tableName, SQLValue[] columnNames,
									SQLCondition condition, SQLOrder[] orders,
									boolean forUpdate) {
		StringBuffer statement = new StringBuffer("SELECT ");

		switch (this.dialect) {
		default:
			statement.append(
					columnNames!=null && columnNames.length > 0?
							columnNameList(columnNames, false) : "*");
			statement.append(" FROM ");
			statement.append(tableName);
			if (condition != null) {
				statement.append(" WHERE ");
				statement.append(condition.toSQL(this.dialect));
			}
			if (orders != null && orders.length > 0) {
				statement.append(" ORDER BY ");
				statement.append(SQLConstruct.commaSeperated(orders).toSQL(this.dialect));
			}
			if (forUpdate) {
				statement.append(" FOR UPDATE");
			}
			break;

		}

		return statement.toString();
	}

	 /**
   * Generates a <code>SELECT DISTINCT</code> statement for the given {@link SQLDialect} as follows:
   * <br><br>
   * For <code>columnNames</code> == <code>null</code>:<br><br>
   * <code>SELECT DISTINCT * FROM tableName [WHERE condition]</code><br>
   * <br>
   * For <code>columnNames</code> != <code>null</code>:<br><br>
   * <code>SELECT DISTINCT tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
   *
   * @param tableName the {@link String} name of the table to be queried
   * @param columnNames the {@link String} names of the columns to fetch
   * @param condition the criteria by which to select or <code>null</code> if no criteria exists
   *
   * @return a {@link String} <code>SELECT DISTINCT</code> statement
   */
  public String selectDistinct(String tableName, SQLValue[] columnNames, SQLCondition condition) {
    return selectDistinct(tableName, columnNames, condition, null);
  }

	/**
	 * Generates a <code>SELECT DISTINCT</code> statement for the given {@link SQLDialect} as follows:
	 * <br><br>
	 * For <code>columnNames</code> == <code>null</code>:<br><br>
	 * <code>SELECT DISTINCT * FROM tableName [WHERE condition]</code><br>
	 * <br>
	 * For <code>columnNames</code> != <code>null</code>:<br><br>
	 * <code>SELECT DISTINCT tableName.columnName[0], tableName.columnName[1], tableName.columName[n] FROM tableName WHERE condition</code>
	 *
	 * @param tableName the {@link String} name of the table to be queried
	 * @param columnNames the {@link String} names of the columns to fetch
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 * @param orders an optional array of {@link SQLOrder} objects
	 *
	 * @return a {@link String} <code>SELECT DISTINCT</code> statement
	 */
	public String selectDistinct(String tableName, SQLValue[] columnNames, SQLCondition condition, SQLOrder[] orders) {
		StringBuffer statement = new StringBuffer("SELECT DISTINCT ");

		switch (this.dialect) {
		default:
			statement.append(
					columnNames!=null && columnNames.length > 0?
							columnNameList(columnNames, false) : "*");
			statement.append(" FROM ");
			statement.append(tableName);
			if (condition != null) {
				statement.append(" WHERE ");
				statement.append(condition.toSQL(this.dialect));
			}
      if (orders != null && orders.length > 0) {
        statement.append(" ORDER BY ");
        statement.append(SQLConstruct.commaSeperated(orders).toSQL(this.dialect));
      }
			break;

		}

		return statement.toString();
	}

	/**
	 * Generates a <code>INSERT</code> statement for the given {@link SQLDialect} as follows:
	 * <br><br>
	 * <code>INSERT INTO tableName (columnName[0], columnName[1], ... columnName[n]) VALUES (?, ?, ... ?)</code>
	 *
	 * @param tableName the {@link String} name of the table to be modified
	 * @param columnNames the {@link String} names of the columns into which to insert
	 * @return A valid <code>INSERT</code> statement.
	 */
	public String insert(String tableName, SQLValue[] columnNames) {
		StringBuffer statement = new StringBuffer("INSERT INTO ");

		switch (this.dialect) {
		default:
			statement.append(tableName);
			statement.append(" (");
			statement.append(unqualifiedColumnNameList(columnNames, false));
			statement.append(") VALUES (");
			statement.append(placeholderList(columnNames.length));
			statement.append(")");
			break;
		}

		return statement.toString();
	}

	/**
	 * Generates a <code>UPDATE</code> statement for a given {@link SQLDialect} as follows:
	 * <br><br>
	 * <code>UPDATE tableName SET columnName[0]=?, columnName[1]=?, columnName[n]=? [WHERE condition]</code>
	 *
	 * @param tableName the {@link String} name of the table to be modified
	 * @param columnNames the {@link String} names of the columns into which to modify
	 * @param condition the criteria by which to select or <code>null</code> if no criteria exists
	 *
	 * @return A valid <code>UPDATE</code> statement.
	 */
	public String update(String tableName, SQLValue[] columnNames, SQLCondition condition) {
		StringBuffer statement = new StringBuffer("UPDATE ");

		switch (this.dialect) {
		default:
			statement.append(tableName);
			statement.append(" SET ");
			statement.append(unqualifiedColumnNameList(columnNames, true));
			if (condition != null) {
				statement.append(" WHERE ");
				statement.append(condition.toSQL(this.dialect));
			}
			break;
		}

		return statement.toString();
	}

	/**
	 * Generates a <code>DELETE</code> statement for a given {@link SQLDialect} as follows:
	 * <br><br>
	 * <code>DELETE FROM tableName [WHERE condition]</code>
	 *
	 * @param tableName the table from which to delete a relation
	 * @param condition the criteria to determine the relation or <code>null</code> if all rows
	 * 			should be deleted
	 * @return A valid <code>DELETE</code> statement.
	 */
	public String delete(String tableName, SQLCondition condition) {
		StringBuffer statement = new StringBuffer("DELETE FROM ");

		switch (this.dialect) {
		default:
			statement.append(tableName);
			if (condition != null) {
				statement.append(" WHERE ");
				statement.append(condition.toSQL(this.dialect));
			}
			break;
		}

		return statement.toString();
	}

	/**
	 * Generates a <code>SELECT/INNER JOIN</code> statement for a given {@link SQLDialect} as follows:
	 * <br><br>
	 * <code>SELECT columnList FROM tableName1 INNER JOIN tableName2 ON joinCondition [WHERE queryCondition]</code><br><br>
	 * Whereby <code>columnList</code> is replaced by the wildcard character '<code>*</code>' if it is <code>null</code>
	 *
	 * @param tableName1 the name of the first (left) table
	 * @param tableName2 the name of the second (right) table
	 * @param joinCondition the {@link SQLCondition} on which to perform the join
	 * @param queryCondition optional {@link SQLCondition} to limit the query further
	 * @param columnNames the columns from which to select or <code>null</code> to insert a wildcard
	 *
	 * @return A valid <code>SELECT/INNER JOIN</code> statement
	 */
	public String innerJoin(String tableName1, String tableName2, SQLCondition joinCondition, SQLCondition queryCondition, SQLValue[] columnNames) {
		StringBuffer statement = new StringBuffer("SELECT ");

		switch (this.dialect) {
		default:
			statement.append(
					columnNames!=null && columnNames.length > 0?
							columnNameList(columnNames, false) : "*");
			statement.append(" FROM ");
			statement.append(tableName1);
			statement.append(" INNER JOIN ");
			statement.append(tableName2);
			statement.append(" ON ");
			statement.append(joinCondition.toSQL(this.dialect));
			if (queryCondition != null) {
				statement.append(" WHERE ");
				statement.append(queryCondition.toSQL(this.dialect));
			}
			break;
		}

		return statement.toString();
	}

	/**
	 * Generates a list of column names from the given {@link String} array in the appropriate dialect.
	 * <br><br>
	 * @param columnNames an array of column names as {@link SQLValue}s
	 * @param placeholders append a {@link PreparedStatement} placeholder value ('=?')
	 *
	 * @return a {@link String} list of names
	 */
	public String columnNameList(SQLValue[] columnNames, boolean placeholders) {

		if (columnNames == null || columnNames.length == 0)
			return "*";

		StringBuffer result = new StringBuffer();

		switch (this.dialect) {
		default:
			for (int i=0; i < columnNames.length; i++) {
				result.append(columnNames[i].toSQL(this.dialect));

				if (placeholders)
					result.append("=").append(SQLValue.INSERT_VALUE.toSQL(this.dialect));

				if (i < columnNames.length - 1)
					result.append(", ");
			}
			break;
		}

		return result.toString();
	}

    /**
     * Generates a list of unqualified column names from the given {@link String} array
     * in the appropriate dialect.
     *
     * @param columnNames an array of column names as {@link SQLValue}s
     * @param placeholders append a {@link PreparedStatement} placeholder value ('=?')
     *
     * @return a {@link String} list of names
     * @see SQLValue#toUnqualifiedSQL(SQLDialect)
     */
    public String unqualifiedColumnNameList(SQLValue[] columnNames, boolean placeholders) {

        if (columnNames == null || columnNames.length == 0)
            return "*";

        StringBuffer result = new StringBuffer();

        switch (this.dialect) {
        default:
            for (int i=0; i < columnNames.length; i++) {
                result.append(columnNames[i].toUnqualifiedSQL(this.dialect));

                if (placeholders)
                    result.append("=").append(SQLValue.INSERT_VALUE.toSQL(this.dialect));

                if (i < columnNames.length - 1)
                    result.append(", ");
            }
            break;
        }

        return result.toString();
    }

    /**
	 * Generates a list of {@link PreparedStatement} placeholders ('?') for the given {@link SQLDialect}<br>
	 *
	 * @param amount the amount of placeholders to put in the list
	 * @return a String of comma-separated <code>PreparedStatement</code> placeholders
	 */
	public String placeholderList(int amount) {
		StringBuffer result = new StringBuffer();

		switch (this.dialect) {
		default:
			for (int i=0; i < amount; i++) {
				result.append(SQLValue.INSERT_VALUE.toSQL(this.dialect));
				if (i < amount - 1)
					result.append(", ");
			}
			break;
		}

		return result.toString();
	}

	/**
	 * @return The SQL dialect for this statement generator
	 */
	public SQLDialect getDialect() {
		return this.dialect;
	}

    /**
     * @param dialect THe SQL dialect to use for this statement generator.
     */
    public void setDialect(SQLDialect dialect) {
        this.dialect = dialect;
    }


}
