package org.clazzes.util.sql.transactionprovider;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.apache.commons.lang.NullArgumentException;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.sql.helper.JDBCHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The implementation of TransactionProvider to be configured via blueprint and DBSetup pattern.
 */
public class TransactionProviderBlueprintImpl implements TransactionProvider
{
    private static final int DEFAULT_ISOLATION_LEVEL = -1; // do not change default

    private static final Logger log = LoggerFactory.getLogger(TransactionProviderBlueprintImpl.class);

    private DataSource dataSource;
    public void setDataSource(DataSource dataSource) { this.dataSource = dataSource; }

    private String threadLocalKey;
    public void setThreadLocalKey(String threadLocalKey) { this.threadLocalKey = threadLocalKey; }

    @Override
    public ClosableTransaction getTransaction() {
        return getTransaction(this.threadLocalKey, DEFAULT_ISOLATION_LEVEL);
    }

    @Override
    public ClosableTransaction getTransaction(int isolationLevel) {
        return getTransaction(this.threadLocalKey, isolationLevel);
    }

    @Override
    public ClosableTransaction getTransaction(String useThreadLocalKey) {
        return getTransaction(useThreadLocalKey, DEFAULT_ISOLATION_LEVEL);
    }

    public ClosableTransaction getTransaction(String useThreadLocalKey, int isolationLevel) {
        if (useThreadLocalKey == null) {
            throw new NullArgumentException("threadLocalKeyOverride");
        }
        Connection connection = getConnection();
        boolean previousAutoCommit = getPreviousAutoCommit(connection);
        int previousIsolationLevel = getPreviousIsolationLevel(connection);
        setNoAutoCommit(connection);
        setIsolationLevel(connection, isolationLevel);

        // bind string to local variable for multithreading
        ThreadLocalManager.bindResource(useThreadLocalKey, connection);
        // the unbindResource is inside TransactionImpl::close()
        return new ClosableTransactionImpl(useThreadLocalKey, previousIsolationLevel, previousAutoCommit);
    }

    private boolean getPreviousAutoCommit(Connection connection) {
        try {
            return connection.getAutoCommit();
        } catch (SQLException e) {
            log.warn("SQLExcption while reading previous autoCommit and isolationLevel", e);
        }
        return true;
    }

    private int getPreviousIsolationLevel(Connection connection) {
        try {
            return connection.getTransactionIsolation();
        } catch (SQLException e) {
            log.warn("SQLExcption while reading previous autoCommit and isolationLevel", e);
        }
        return Connection.TRANSACTION_READ_COMMITTED;
    }

    private void setIsolationLevel(Connection connection, int isolationLevel) {
        try {
            if (isolationLevel != DEFAULT_ISOLATION_LEVEL) {
                if (log.isDebugEnabled()) {
                    log.debug("Starting JDBC transaction with isolation level [{}].",
                            JDBCHelper.formatIsolationLevel(isolationLevel));
                }

                int originalIsolationLevel = connection.getTransactionIsolation();
                if (originalIsolationLevel != isolationLevel) {
                    connection.setTransactionIsolation(isolationLevel);
                }
            }
        } catch (SQLException e) {
            throw new TransactionException("Error while setting isolation level.", e);
        }
    }

    @Override
    public boolean isActive() {
        // the datasource is set by bundle activators
        return this.dataSource != null;
    }

    private Connection getConnection() {
        Connection connection;
        try {
            connection = this.dataSource.getConnection();
        } catch (SQLException e) {
            throw new TransactionException("Error while attempting to obtain a connection.", e);
        }
        return connection;
    }

    private void setNoAutoCommit(Connection connection) {
        try {
            connection.setAutoCommit(false);
        } catch (SQLException e) {
            throw new TransactionException("Error while disabling autocommit.", e);
        }
    }
}
