package org.clazzes.util.sql.transactionprovider;

import java.util.function.Supplier;

/**
 * Provide your code with a transaction in the configured data source binding the connection on to a thread local key.
 *
 * Note: You need to Re-/Set the dataSource as it got up.
 */
public interface TransactionProvider {
    /**
     * Call action within a (new) transaction.
     *
     * @throws TransactionException If the transaction got a problem.
     */
    default <T> T transacted(Supplier<T> action) throws TransactionException {
        try (ClosableTransaction autoClose = getTransaction()) {
            T result = action.get();
            autoClose.willCommit();
            return result;
        }
	}

    /**
     * Getting an AutoClosable transaction. Uses the default thread local key.
     * Remark: You need to call result.willCommit() explicitly.
     * @return AutoClosable, use with try-with-resource!
     */
    ClosableTransaction getTransaction();

    /**
     * Gets a AutoClosable transaction with the given isolation level using the default thread local key.
     * Remark: You need to call result.willCommit() explicitly.
     * @return AutoClosable, use with try-with-resource!
     */
    ClosableTransaction getTransaction(int isolationLevel);

    /**
     * Getting an AutoClosable transaction.
     * Remark: You need to call result.willCommit() explicitly.
     * @param useThreadLocalKey The thread local key to use.
     * @return AutoClosable, use with try-with-resource!
     */
    ClosableTransaction getTransaction(String useThreadLocalKey);

    /**
     * Getting an AutoClosable transaction.
     * Remark: You need to call result.willCommit() explicitly.
     * @param useThreadLocalKey The thread local key to use.
     * @param isolationLevel The isolation level to start in.
     * @return AutoClosable, use with try-with-resource!
     */
    ClosableTransaction getTransaction(String useThreadLocalKey, int isolationLevel);

    /**
     * The datasource is set (i.e. exists)
     */
    boolean isActive();
}
