package org.clazzes.util.sql.helper;

import java.io.UnsupportedEncodingException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.clazzes.persistence.api.dto.IPersistent;
import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.jdbc.JdbcDAOSupport;
import org.clazzes.util.sql.dao.IIdDAO;
import org.clazzes.util.sql.dao.StatementPreparer;

public class QueryHelper {
    @SuppressWarnings("unused")
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(QueryHelper.class);


	public static String getPlaceHolderSequence(int numberOfPlaceHolders) {
        if (numberOfPlaceHolders == 0) {
            return "NULL";
        }
		String placeHolderString = "";
		for (int n = 0; n < numberOfPlaceHolders; n++) {
			placeHolderString += "?" + (n < numberOfPlaceHolders - 1 ? ", " : "");
		}
		return placeHolderString;
	}

	public static int setIdsToStatement(PreparedStatement statement, int currIndex, List<Long> ids) throws SQLException {
		for (Long id : ids) {
			JDBCHelper.setLong(statement, currIndex++, id);
		}
		return currIndex;
	}

	public static int getNumberOfWildcards(String sql) {
		int numberOfWildcards = 0;
		int currIndex = 0;
		while (true) {
			int position = sql.indexOf("?", currIndex);
			if (position == -1) {
				break;
			} else {
				currIndex = position + 1;
				numberOfWildcards++;
			}
		}
		return numberOfWildcards;
	}

	public static void checkNumberOfWildcards(String sql, int currIndex) {
		int numberOfWildcards = QueryHelper.getNumberOfWildcards(sql);
		if (numberOfWildcards != currIndex - 1) {
			throw new RuntimeException("Total number of wildcards: " + numberOfWildcards + "; number of set wildcards: " + (currIndex - 1)
										+ "; please find the bug.");
		}
	}

	public static <T,L> List<T> executeDAOGetter(List<L> ids, int batchSize, BatchedDAOGetter<T,L> getter) {
    	List<T> retList = new ArrayList<T>();
    	int currOffset = 0;
    	while (currOffset < ids.size()) {
    		int delta = batchSize;
    		List<L> currIds = ids.subList(currOffset, Math.min(currOffset + delta, ids.size()));
    		retList.addAll(getter.execute(currIds));
    		currOffset += delta;
    	}
    	return retList;
	}

	public static <T,L> Map<L,T> executeDAOGetter(List<L> ids, int batchSize, BatchedDAOGetterMap<T,L> getter) {
        Map<L,T> retMap = new HashMap<L,T>();
        int currOffset = 0;
        while (currOffset < ids.size()) {
            int delta = batchSize;
            List<L> currIds = ids.subList(currOffset, Math.min(currOffset + delta, ids.size()));
            retMap.putAll(getter.execute(currIds));
            currOffset += delta;
        }
        return retMap;
	}
	
	public static <L> void executeBatch(List<L> ids, int batchSize, BatchedExecutor<L> executor) {
        int currOffset = 0;
        while (currOffset < ids.size()) {
            int delta = batchSize;
            List<L> currIds = ids.subList(currOffset, Math.min(currOffset + delta, ids.size()));
            executor.execute(currIds);
            currOffset += delta;
        }
	}

	public static byte[] encodeStringForDatabaseBlob(String s) {
		try {
			return s.getBytes("UTF-8");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}
	}

	public static List<Long> getPersistentIds(List<? extends IPersistent> persistents) {
	    List<Long> persistentIds = new ArrayList<Long>();
	    for (IPersistent persistent : persistents) {
	        persistentIds.add(persistent.getId());
	    }
	    return persistentIds;
	}
	
	public static Set<Long> getPersistentIdsSet(List<? extends IPersistent> persistents) {
        Set<Long> persistentIds = new HashSet<Long>();
        for (IPersistent persistent : persistents) {
            persistentIds.add(persistent.getId());
        }
        return persistentIds;	    
	}

	public static String trimOptionalFieldValue(String s) {
        s = s != null ? s.trim() : null;
        if (s == null || s.length() == 0) {
            return null;
        } else {
            return s;
        }
	}

	public static String trimMandatoryFieldValue(String s, String fieldName) {
	    s = s != null ? s.trim() : null;
	    if (s == null || s.length() == 0) {
	        throw new IllegalArgumentException("Field '" + fieldName + "' is mandatory, but missing.  Please provide a value.");
	    } else {
	        return s;
	    }
	}

	public static <T> T checkMandatoryValue(T value, String fieldName) {
	    if (value == null) {
	        throw new IllegalArgumentException("Field '" + fieldName + "' is mandatory, but missing. Please provider a value.");
	    } else {
	        return value;
	    }
	}

    @FunctionalInterface
    public static interface StatementPreparerUsingFiller {
        public void prepare(StatementFiller statementFiller) throws SQLException;
    }

    @FunctionalInterface
    public static interface ResultSetMapper<T> {
        public T map(ResultSet rs) throws SQLException;
    }

    public static <T> List<T> getListWithSql(JdbcDAOSupport dao, String sql, ResultSetMapper<T> mapper, StatementPreparer preparer) {
        Connection con =  ThreadLocalManager.getBoundResource(dao.getThreadLocalKey());

        PreparedStatement statement = null;

        try {
            statement = con.prepareStatement(sql);

            preparer.fillInsertValues(statement);

            List<T> ret = new ArrayList<T>();

            ResultSet rs = statement.executeQuery();

            while (rs.next()) {
                ret.add(mapper.map(rs));
            }

            return ret;
        } catch (Exception e) {
            throw new DAOException(e);
        } finally {
            if (statement != null) {
                try {
                    statement.close();
                } catch (Exception e) {
                    log.warn("Error when closing prepared statement.", e);
                }
            }
        }
    }

    public static <T> List<T> getListWithSqlFiller(JdbcDAOSupport dao, String sql, ResultSetMapper<T> mapper, StatementPreparerUsingFiller preparer) {
        return getListWithSql(dao, sql, mapper, statement -> {
                preparer.prepare(new StatementFiller(statement));
                });
    }

    public static <T> T getUniqueWithSql(JdbcDAOSupport dao, String sql, ResultSetMapper<T> mapper, StatementPreparer preparer) {
        Connection con =  ThreadLocalManager.getBoundResource(dao.getThreadLocalKey());

        PreparedStatement statement = null;

        try {
            statement = con.prepareStatement(sql);

            preparer.fillInsertValues(statement);

            T ret = null;

            ResultSet rs = statement.executeQuery();

            while (rs.next()) {
                if (ret != null) {
                    throw new DAOException("getUniqueWithCondition found multiple "
                                           + " instances of [" + ret.getClass().getSimpleName() + "] where at most one was expected.");

                } else {
                ret = mapper.map(rs);
                }
            }

            return ret;
        } catch (Exception e) {
            throw new DAOException(e);
        } finally {
            if (statement != null) {
                try {
                    statement.close();
                } catch (Exception e) {
                    log.warn("Error when closing prepared statement.", e);
                }
            }
        }
    }

    // from MDA
	/** Synchronizes the database state given in the oldPersistents list
	 *  with the expected state given by the newPersistents List, such that
	 *  afterwards exactly the instances from the newPersistents List exist
	 *  at the database.
	 *
	 *  The Persistents are matched by their ids, no equals comparisons are
	 *  performed.
	 *
	 *  Instances from newPersistents with id == null are saved, with id != null
	 *  updated if they have a counterpart in oldPersistents.
	 *
	 *  If there is no such counterparts, their old ids are removed, and they
	 *  are saved newly; i.e. we really expect all oldPersistents in the parameter
	 *  list.  
	 *  
	 *  Note that we rely on the aforementioned behaviour if the client generates
	 *  new Persistents with artificial (often negative) ids.  This is e.g. a
	 *  common case in the context of ApuGrid code, where the grid needs an
	 *  id for all instances regardless wether they have a database counterpart
	 *  or not.
	 *
	 *  Instances present in oldPersistents, but not in newPersistents
	 *  (again by id), are deleted.
	 *
	 * @param <T> type of the Persistents
	 * @param dao the corresponding DAO
	 * @param newPersistents expected state of database afterwards, as described
	 * @param oldPersistents old database state, as described
	 */
    public static <T extends IPersistent> SyncWithDBResult<T> syncWithDatabase(IIdDAO<T> dao, List<T> newPersistents,
                                                                  List<T> oldPersistents) {
        return QueryHelper.syncWithDatabase(dao, newPersistents, oldPersistents, false);
    }


    public static <T extends IPersistent> SyncWithDBResult<T> syncWithDatabase(IIdDAO<T> dao, List<T> newPersistents,
                                                                  List<T> oldPersistents, boolean skipDelete) {
		List<T> persistentsToSave = new ArrayList<T>();
		List<T> persistentsToUpdate = new ArrayList<T>();
		List<Long> persistentIdsToDelete = new ArrayList<Long>();

		Set<Long> oldPersistentIds = new HashSet<Long>();
		for (T oldPersistent : oldPersistents) {
			oldPersistentIds.add(oldPersistent.getId());
		}

		Set<Long> newPersistentIds = new HashSet<Long>();
		List<T> retValues = new ArrayList<T>();
		Map<Integer, Integer> toSaveIndexToRetIndex = new HashMap<Integer, Integer>();
		for (T newPersistent : newPersistents) {
			Long id = newPersistent.getId();
			if (id != null && !oldPersistentIds.contains(id)) {
				// This case might refer to artificial ids living only
				// at client side, or to an old client state
				// (if e.g. someone else deleted the object one edits).
				// In either case, save the state the client transmits,
				// instead of throwing an Exception.
				newPersistent.setId(null);
                toSaveIndexToRetIndex.put(persistentsToSave.size(), retValues.size());
				persistentsToSave.add(newPersistent);
				retValues.add(newPersistent);
			} else if (id == null) {
                toSaveIndexToRetIndex.put(persistentsToSave.size(), retValues.size());
				persistentsToSave.add(newPersistent);
				retValues.add(newPersistent);
			} else {
				persistentsToUpdate.add(newPersistent);
				newPersistentIds.add(id);
				retValues.add(newPersistent);
			}
		}

		// Delete all instances that are not marked to stay
		// according to the newPersistents list.
		for (T oldPersistent : oldPersistents) {
			Long id = oldPersistent.getId();
			if (!newPersistentIds.contains(id)) {
				persistentIdsToDelete.add(id);
			}
		}

		if (!persistentsToSave.isEmpty()) {
			List<T> savedPersistents = dao.saveBatch(persistentsToSave);
			for (Integer toSaveIndex : toSaveIndexToRetIndex.keySet()) {
			    Integer retIndex = toSaveIndexToRetIndex.get(toSaveIndex);
			    retValues.set(retIndex, savedPersistents.get(toSaveIndex));
			}
		}
		if (!persistentsToUpdate.isEmpty()) {
			dao.updateBatch(persistentsToUpdate);
		}
		
		SyncWithDBResult<T> result = new SyncWithDBResult<T>();
		if (!skipDelete) {
	        if (!persistentIdsToDelete.isEmpty()) {
	            try {
	                dao.deleteBatch(persistentIdsToDelete);
	            } catch (Exception e) {
	                // Find out which id is non-deletable, and register it
	                for (Long persistentIdToDelete : persistentIdsToDelete) {
	                    try {
	                        dao.delete(persistentIdToDelete);
	                    } catch (Exception e2) {
	                        result.addNotDeletableId(persistentIdToDelete);
	                    }
	                }
	            }
	        }
		}
		
		result.setInstances(retValues);
		return result;
	}
    
    public static <T> void syncWithDatabaseByMap(IIdDAO<T> dao, Map<Long, T> idToNewItem,
                                                 Map<Long, T> idToOldItem) {
        QueryHelper.syncWithDatabaseByMap(dao, idToNewItem, idToOldItem, false);
    }
    
    
    public static <T> void syncWithDatabaseByMap(IIdDAO<T> dao, Map<Long, T> idToNewItem,
                                                 Map<Long, T> idToOldItem, boolean skipDelete) {
        List<T> saveItems = new ArrayList<T>();
        List<Long> deleteIds = new ArrayList<Long>();
        List<T> updateItems = new ArrayList<T>();
        
        for (Long id : idToNewItem.keySet()) {
            T item = idToNewItem.get(id);
            if (idToOldItem.containsKey(id)) {
                updateItems.add(item);
            } else {
                saveItems.add(item);
            }
        }
        
        for (Long id : idToOldItem.keySet()) {
            if (!idToNewItem.containsKey(id)) {
                deleteIds.add(id);
            }
        }
        
        if (saveItems.size() > 0) {
            dao.saveBatch(saveItems);
        }
        if (updateItems.size() > 0) {
            dao.updateBatch(updateItems);
        }
        if (!skipDelete) {
            if (deleteIds.size() > 0) {
                dao.deleteBatch(deleteIds);
            }            
        }
    }

    public static <T> T getUniqueWithSqlFiller(JdbcDAOSupport dao, String sql, ResultSetMapper<T> mapper, StatementPreparerUsingFiller preparer) {
        return getUniqueWithSql(dao, sql, mapper, statement -> {
                preparer.prepare(new StatementFiller(statement));
                });
    }
    
    public static String getLengthFunctionName(JdbcDAOSupport dao) {
        Connection con = ThreadLocalManager.getBoundResource(dao.getThreadLocalKey());
        try {
            if ("Microsoft SQL Server".equals(con.getMetaData().getDatabaseProductName())) {
                return "LEN";
            } else {
                return "LENGTH";
            }
        } catch (SQLException e) {
            log.warn("Error when trying to determine database product.", e);
        }
        return "LENGTH";
    }

    public static String getLimitClause(JdbcDAOSupport dao, String number, boolean hasOrderBy) {
        Connection con = ThreadLocalManager.getBoundResource(dao.getThreadLocalKey());

        try {
            if ("Microsoft SQL Server".equals(con.getMetaData().getDatabaseProductName())) {
                if (hasOrderBy) {
                    return ""
                        + "OFFSET 0 ROWS\n"
                        + "FETCH NEXT " + number + " ROWS ONLY\n"
                        + "";
                } else {
                    return ""
                        + "ORDER BY 0\n"
                        + "OFFSET 0 ROWS\n"
                        + "FETCH NEXT " + number + " ROWS ONLY\n"
                        + "";
                }
            }
        } catch (SQLException e) {
            log.warn("Error when trying to determine database product.", e);
        }

        return "LIMIT " + number + "\n";
    }

    public static String getLimitClause(JdbcDAOSupport dao, String number) {
        return getLimitClause(dao, number, true);
    }
}
