/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.helper;

import java.sql.Connection;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An auto-closeable transaction, which imposes a given
 * isolation level for usage with java7's try-with-resources
 */
public class JDBCTransaction implements AutoCloseable {

    private static final Logger log = LoggerFactory.getLogger(JDBCTransaction.class);
    
    private final int isolationLevel;
    private final boolean doCloseConnection;
    private Connection connection;
    private int originalIsolationLevel;
    
    protected void startTransaction() throws SQLException {
        
        if (this.isolationLevel != -1) {
            
            if (log.isDebugEnabled()) {
                log.debug("Starting JDBC transaction with isolation level [{}].",
                        JDBCHelper.formatIsolationLevel(this.isolationLevel));
            }

            this.originalIsolationLevel = this.connection.getTransactionIsolation();
            
            if (this.originalIsolationLevel != this.isolationLevel) {
                
                this.connection.setTransactionIsolation(this.isolationLevel);
            }
        }
        else {
            
            if (log.isDebugEnabled()) {
                log.debug("Starting JDBC transaction with default isolation level.");
            }
        }
        
        this.connection.setAutoCommit(false);
    }
    
    protected void dropConnection() {
        
        if (this.originalIsolationLevel != this.isolationLevel) {
            try {
                this.connection.setTransactionIsolation(this.originalIsolationLevel);
            } catch (SQLException e) {
                log.warn("Error resetting isolation level from ["+
                        JDBCHelper.formatIsolationLevel(this.isolationLevel)+"] to ["+
                        JDBCHelper.formatIsolationLevel(this.originalIsolationLevel)+"]",e);
            }
        }
        
        if (this.doCloseConnection) {
            try {
                this.connection.close();
            } catch (SQLException e) {
                log.warn("Error closing JDBC connection",e);
            }
        }
        
        this.connection = null;
    }
    
    protected static Connection fetchConnection(DataSource dataSource) throws SQLException {
        
        Connection connection;
        
        if (log.isTraceEnabled()){
            
            long now = System.currentTimeMillis();
            log.trace("Fetching connection from datasource [{}].",dataSource);
            
            connection = dataSource.getConnection();
            
            long duration = System.currentTimeMillis() - now;
            log.trace("Opening transaction for datasource [{}], fetch took [{}] milliseconds.",dataSource,duration);
        }
        else {
            
            connection = dataSource.getConnection();
        }
        return connection;
    }
    
    /**
     * Pull a connection from the given datasource and open a transaction
     * with the default isolation level of the datasource.
     * The pulled connection will be closed inside {@link #close()}.
     * 
     * @param dataSource The data source to pull a connection from.
     * @throws SQLException Upon errors opening the transaction.
     */
    public JDBCTransaction(DataSource dataSource) throws SQLException {
        this.connection = fetchConnection(dataSource);
        this.doCloseConnection = true;
        this.isolationLevel = -1;
        this.originalIsolationLevel = -1;
        this.startTransaction();
    }
    
    /**
     * Pull a connection from the given datasource and open a transaction
     * with the given isolation level.
     * The pulled connection will be closed inside {@link #close()}.
     * 
     * @param dataSource The data source to pull a connection from.
     * @param isolationLevel The isolation level to use.
     * @throws SQLException Upon errors opening the transaction.
     * 
     * @see Connection#TRANSACTION_NONE
     * @see Connection#TRANSACTION_READ_UNCOMMITTED
     * @see Connection#TRANSACTION_READ_COMMITTED
     * @see Connection#TRANSACTION_REPEATABLE_READ
     * @see Connection#TRANSACTION_SERIALIZABLE
     */
    public JDBCTransaction(DataSource dataSource, int isolationLevel) throws SQLException {
        this.connection = fetchConnection(dataSource);
        this.doCloseConnection = true;
        this.isolationLevel = isolationLevel;
        this.startTransaction();
    }
        
    /**
     * Open a transaction on the given connection using the default
     * isolation level of the connection. 
     * The pulled connection will not be closed inside {@link #close()}.
     * 
     * @param connection The JDBC connection to operate on.
     * @throws SQLException Upon errors opening the transaction.
     */
    public JDBCTransaction(Connection connection) throws SQLException {
        this.connection = connection;
        this.doCloseConnection = false;
        this.isolationLevel = -1;
        this.originalIsolationLevel = -1;
        this.startTransaction();
    }
    
    /**
     * Open a transaction on the given connection using the given
     * isolation level. 
     * The pulled connection will not be closed inside {@link #close()}.
     * 
     * @param connection The JDBC connection to operate on.
     * @param isolationLevel The isolation level to use.
     * @throws SQLException Upon errors opening the transaction.
     * 
     * @see Connection#TRANSACTION_NONE
     * @see Connection#TRANSACTION_READ_UNCOMMITTED
     * @see Connection#TRANSACTION_READ_COMMITTED
     * @see Connection#TRANSACTION_REPEATABLE_READ
     * @see Connection#TRANSACTION_SERIALIZABLE
     */
    public JDBCTransaction(Connection connection, int isolationLevel) throws SQLException {
        this.connection = connection;
        this.doCloseConnection = false;
        this.isolationLevel = isolationLevel;
        this.startTransaction();
    }

    /**
     * @return Whether this transaction is currently active. Returns <code>false</code>
     *         after {@link #commit()}, {@link #rollback()} or {@link #clone()}.
     */
    public boolean isOpen() {
        return this.connection != null;
    }
    
    /**
     * @return The connection we are currently operating on or <code>null</code>,
     *         if this transaction has already been rolled back or committed.
     */
    public Connection getConnection() {
        return this.connection;
    }
    
    /**
     * Commit the transaction.
     * 
     * @throws SQLException Upon errors.
     */
    public void commit() throws SQLException {
        
        if (log.isDebugEnabled()) {
            log.debug("Committing JDBC transaction.");
        }
        
        this.connection.commit();
        this.dropConnection();
    }

    /**
     * An alias for {@link #close()}.
     * 
     * @throws SQLException Upon errors rolling back the transaction.
     */
    public void rollback() throws SQLException {
        this.close();
    }
    
    /**
     * Roll back the transaction.
     * @throws SQLException Upon errors rolling back the transaction.
     */
    @Override
    public void close() throws SQLException {
        
        if (this.connection != null) {
            
            if (log.isDebugEnabled()) {
                log.debug("Rolling back JDBC transaction.");
            }
            
            try {
                this.connection.rollback();
            }
            finally {
                this.dropConnection();
            }
        }
    }
    
}
