/* **********************************************************
 * $Id$
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql.helper;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Locale;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Pattern;

import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.i18n.I18nString;

/**
 * <p>Static helper functions for Number object binding/fetching
 * from JDBC result sets and prepared statements.</p>
 *
 * <p>These functions greatly ease the life
 * of programmers when working with
 * {@link Long}, {@link Integer},... properties, because the
 * plain JDBC API is very tedious to use.</p>
 */
public abstract class JDBCHelper {

    /**
     * Format the transaction isolation level constants as a string.
     *
     * @param isolationLevel A transaction isolation level constant.
     * @return A string representation.
     *
     * @see Connection#TRANSACTION_NONE
     * @see Connection#TRANSACTION_READ_UNCOMMITTED
     * @see Connection#TRANSACTION_READ_COMMITTED
     * @see Connection#TRANSACTION_REPEATABLE_READ
     * @see Connection#TRANSACTION_SERIALIZABLE
     */
    public static final String formatIsolationLevel(int isolationLevel) {

        switch (isolationLevel) {

        case Connection.TRANSACTION_NONE:
            return "TRANSACTION_NONE";

        case Connection.TRANSACTION_READ_UNCOMMITTED:
            return "TRANSACTION_READ_UNCOMMITTED";

        case Connection.TRANSACTION_READ_COMMITTED:
            return "TRANSACTION_READ_COMMITTED";

        case Connection.TRANSACTION_REPEATABLE_READ:
            return "TRANSACTION_REPEATABLE_READ";

        case Connection.TRANSACTION_SERIALIZABLE:
            return "TRANSACTION_SERIALIZABLE";

        default:
            return "TRANSACTION_ISOLATION_UNKNOWN" + isolationLevel;
        }
    }


	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Boolean object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Boolean getBoolean(ResultSet rs, int columnIndex) throws SQLException {

		boolean b = rs.getBoolean(columnIndex);
		return rs.wasNull() ? null : Boolean.valueOf(b);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Boolean object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Boolean getBoolean(ResultSet rs, String columnLabel) throws SQLException {

		boolean b = rs.getBoolean(columnLabel);
		return rs.wasNull() ? null : Boolean.valueOf(b);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Byte object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Byte getByte(ResultSet rs, int columnIndex) throws SQLException {

		byte b = rs.getByte(columnIndex);
		return rs.wasNull() ? null : Byte.valueOf(b);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Byte object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Byte getByte(ResultSet rs, String columnLabel) throws SQLException {

		byte b = rs.getByte(columnLabel);
		return rs.wasNull() ? null : Byte.valueOf(b);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Short object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Short getShort(ResultSet rs, int columnIndex) throws SQLException {

		short s = rs.getShort(columnIndex);
		return rs.wasNull() ? null : Short.valueOf(s);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Short object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Short getShort(ResultSet rs, String columnLabel) throws SQLException {

		short s = rs.getShort(columnLabel);
		return rs.wasNull() ? null : Short.valueOf(s);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return An Integer object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Integer getInt(ResultSet rs, int columnIndex) throws SQLException {

		int i = rs.getInt(columnIndex);
		return rs.wasNull() ? null : Integer.valueOf(i);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return An Integer object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Integer getInt(ResultSet rs, String columnLabel) throws SQLException {

		int i = rs.getInt(columnLabel);
		return rs.wasNull() ? null : Integer.valueOf(i);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Long object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Long getLong(ResultSet rs, int columnIndex) throws SQLException {

		long l = rs.getLong(columnIndex);
		return rs.wasNull() ? null : Long.valueOf(l);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Long object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Long getLong(ResultSet rs, String columnLabel) throws SQLException {

		long l = rs.getLong(columnLabel);
		return rs.wasNull() ? null : Long.valueOf(l);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Float object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Float getFloat(ResultSet rs, int columnIndex) throws SQLException {

		float f = rs.getFloat(columnIndex);
		return rs.wasNull() ? null : Float.valueOf(f);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Float object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Float getFloat(ResultSet rs, String columnLabel) throws SQLException {

		float f = rs.getFloat(columnLabel);
		return rs.wasNull() ? null : Float.valueOf(f);
	}

	/**
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 *                    the first column has the index 1.
	 * @return A Double object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Double getDouble(ResultSet rs, int columnIndex) throws SQLException {

		double d = rs.getDouble(columnIndex);
		return rs.wasNull() ? null : Double.valueOf(d);
	}

	/**
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @return A Double object or <code>null</code>, if
	 *              {@link ResultSet#wasNull()} returned <code>true</code>.
	 * @throws SQLException Upon errors.
	 */
	public static Double getDouble(ResultSet rs, String columnLabel) throws SQLException {

		double d = rs.getDouble(columnLabel);
		return rs.wasNull() ? null : Double.valueOf(d);
	}

	/**
	 * Gets the string of columnIndex (or null).
	 * @param rs A result set.
	 * @param columnLabel The column label of the column to get.
	 * @throws SQLException Upon errors.
	 */
	public static String getString(ResultSet rs, String columnLabel) throws SQLException {
		return rs.getString(columnLabel);
	}

	/**
	 * Gets the string of columnIndex (or null).
	 * @param rs A result set.
	 * @param columnIndex The column index of the column to get;
	 * 	                  the first column has the index 1.
	 * @throws SQLException Upon errors.
	 */
	public static String getString(ResultSet rs, int columnIndex) throws SQLException {
		return rs.getString(columnIndex);
	}


	/** Helper function for the case, that utc seconds are stored in Java code in an attribute of type Double,
     *  and mapped to a database column of type e.g. date, datetime or timestamp.  Reads a java.sql.Date from
     *  the given ResultSet, and transforms it into utcSeconds stored in a Double object.
     *  WARNING: The result depends on the time zone of the JVM.
     *  @param resultSet the resultSet
     *  @param columnIndex the column index of the column to get; the first column has the index 1.
     *  @return the utcSeconds read from the ResultSet, null if the database entry was null
     */
	public static Double getUtcSeconds(ResultSet resultSet, int columnIndex) throws SQLException {
	    Timestamp timeStamp = resultSet.getTimestamp(columnIndex);
	    return timeStamp == null ? null : Double.valueOf(timeStamp.getTime() * 0.001);
    }

	/**
	 * Gets an sql TIMESTAMP value as an Instant with full nanosecond precision.
	 * @param resultSet A result set.
	 * @param columnIndex The column index of the column to get;
	 * 	                  the first column has the index 1.
	 * @param timeZone If the database timestamp does not store a timezone
	 *                    the timestamp will be interpreted as being inside of this timezone
	 * @throws SQLException Upon errors.
	 * @return The converted Instant or null.
	 */
	public static Instant getInstant(ResultSet resultSet, int columnIndex, ZoneId timeZone) throws SQLException {
		Timestamp timeStamp = resultSet.getTimestamp(columnIndex, timeZone == null ? null : Calendar.getInstance(TimeZone.getTimeZone(timeZone)));
		if (timeStamp == null) {
			return null;
		}
		return timeStamp.toInstant();
	}

	public static Instant getInstant(ResultSet resultSet, int columnIndex) throws SQLException {
		return getInstant(resultSet, columnIndex, null);
	}

	private static final String I18N_STRING_PATTERN_TRANSLATION = "([^\"]|\\\\\")*";
	private static final String I18N_STRING_PATTERN_MAPPING = "(\\s*\"[a-z]+\"\\s*:\\s*\"" + I18N_STRING_PATTERN_TRANSLATION + "\"\\s*)";
	private static final Pattern I18N_STRING_PATTERN = Pattern.compile("\\s*\\{" + I18N_STRING_PATTERN_MAPPING + "?\\s*"
	                                                                    + "(\\," + I18N_STRING_PATTERN_MAPPING + "\\s*)*\\}\\s*");

	/** Returns wether the given String is a valid I18nString serialization, something like {"de" : "Deutsche Übersetzung", "en" : "English translation" }.
     *  @param s any String
     *  @return wether s is a valid I18nString serialization
	 */
	public static boolean isI18nMapString(String s) {
	    return I18N_STRING_PATTERN.matcher(s).matches();
	}

	/** Parses the given databaseString as I18nString.
	 * @param databaseString not null, and isI18nMapString(databaseString) returns true
	 * @return the parsed I18nString representation
	 * @throws IllegalArgumentException if string is malformed (no complete check for each kind of malformedness), or
	 *              if some language occurs multiple times
	 */
	public static I18nString parseI18nString(String databaseString) {
	    I18nString retString = new I18nString();

	    // Look up the open and close brackets, containing all the data we are interested in
	    int openBracketPosition = databaseString.indexOf("{");
        int closeBracketPosition = databaseString.lastIndexOf("}");

        if (openBracketPosition == -1 || closeBracketPosition == -1) {
            // In fact the regexp in isI18nMapString should catch this case, but checking here either is no harm
            throw new IllegalArgumentException("Could not determine open/close bracket position in string [" + databaseString + "]; why didn't the RegExp check catch this?");
        } else {

            // Loop once per entry \"de\" : \"Übersetzung\"
            int currPos = openBracketPosition;
            while (currPos != -1 && currPos < closeBracketPosition) {
                // Find out where the language (the key in json sense) is located
                int languageQuotationMarkOpenPos = databaseString.indexOf("\"", currPos);
                int languageQuotationMarkClosePos = databaseString.indexOf("\"", languageQuotationMarkOpenPos + 1);

                // Special case: If no language at all is defined, then we need to exit here.  This is the case databaseString.equals("{}")
                if (languageQuotationMarkOpenPos == -1 || languageQuotationMarkClosePos == -1) {
                    break;
                }

                // Extract language
                String language = databaseString.substring(languageQuotationMarkOpenPos + 1, languageQuotationMarkClosePos);

                // Find colon, and opening quotation mark of the translation
                int colonPos = databaseString.indexOf(":", languageQuotationMarkClosePos);
                int textQuotationMarkOpenPos = databaseString.indexOf("\"", colonPos + 1);
                int currTextPos = textQuotationMarkOpenPos;
                // Find closing quotation mark of the translation; skip quoted quotation marks
                int textQuotationMarkEndPos = -1;
                do {
                    int backslashPos = databaseString.indexOf("\\", currTextPos + 1);
                    textQuotationMarkEndPos = databaseString.indexOf("\"", currTextPos + 1);

                    // If a backslash occurs immediately before a quotation mark, then we have a quoted quotation mark, and need to
                    // continue searching for the closing quotation mark.  Otherwise, finding a quotation mark is sufficient for
                    // terminating the search for the translation bounds.
                    if (backslashPos != -1 && textQuotationMarkEndPos != -1 && backslashPos == textQuotationMarkEndPos - 1) {
                        currTextPos = textQuotationMarkEndPos;
                    } else {
                        break;
                    }
                } while (true);

                // Extract translation
                String translation = databaseString.substring(textQuotationMarkOpenPos + 1, textQuotationMarkEndPos);

                // Catch case that multiple entries for the same language exist
                if (retString.containsLanguage(language)) {
                    throw new IllegalArgumentException("Parsed I18nString contains multiple translations for language [" + language + "]; one of the translations is [" + translation + "]; input string is [" + databaseString + "]");
                }

                retString.setTranslation(language, translation);

                // Jump to the comma leading to the next key/value entry
                currPos = databaseString.indexOf(",",  textQuotationMarkEndPos);
            }
        }

        return retString;
	}

	/** Returns an I18nString based on the value for the given column in the given ResultSet
	 *  (which is expected to be a string).
	 *  Three cases: For null strings, return null.  For non-null, plain strings, return a I18nString
	 *  with exactly one entry, which is the failback language (see ThreadLocalManager) mapped to the
	 *  plain string as translation.  For strings of the syntax { "de" : "Übersetzung", "en" : "Translation" }
	 *  return an I18nString containing the corresponding syntax.
	 *
	 *  For the decision which strings are valid, see I18N_STRING_PATTERN above.  Any string not matching
	 *  that RegExp is considered a plain string in the sense of this comment, i.e. any such string is
	 *  mapped to the failback language.
	 *
	 * @param resultSet ResultSet
	 * @param columnIndex column
	 * @return I18nString or null as described
	 * @throws SQLException if something goes wrong
	 */
	public static I18nString getI18nString(ResultSet resultSet, int columnIndex) throws SQLException {
	    String databaseString = resultSet.getString(columnIndex);
	    if (databaseString == null) {
	        return null;
	    } else if (isI18nMapString(databaseString)) {
	        return parseI18nString(databaseString);
	    } else {
	        Locale fallbackLocale = ThreadLocalManager.getFallbackLocale();

	        if (fallbackLocale == null) {
	            throw new DAOException("Cannot read string [" + databaseString + "] as I18nString, since it has not the correct format; "
	                                 + "additionally the fallback locale is missing.");
	        } else {
	            I18nString retString = new I18nString();
	            retString.setTranslation(fallbackLocale.getLanguage(), databaseString);
	            return retString;
	        }
	    }
	}

	/**
	 * Call <code>ps.setBoolean(parameterIndex,b.booleanValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.BOOLEAN)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param b A Boolean object.
	 * @throws SQLException Upon errors.
	 */
	public static void setBoolean(PreparedStatement ps, int parameterIndex, Boolean b) throws SQLException {

		if (b == null) {
			ps.setNull(parameterIndex,Types.BOOLEAN);
		}
		else {
			ps.setBoolean(parameterIndex,b.booleanValue());
		}
	}

	/**
	 * Call <code>ps.setByte(parameterIndex,b.byteValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.TINYINT)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param b A Byte object.
	 * @throws SQLException Upon errors.
	 */
	public static void setByte(PreparedStatement ps, int parameterIndex, Number b) throws SQLException {

		if (b == null) {
			ps.setNull(parameterIndex,Types.TINYINT);
		}
		else {
			ps.setByte(parameterIndex,b.byteValue());
		}
	}

	/**
	 * Call <code>ps.setShort(parameterIndex,s.shortValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.SMALLINT)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param s A Short object.
	 * @throws SQLException Upon errors.
	 */
	public static void setShort(PreparedStatement ps, int parameterIndex, Number s) throws SQLException {

		if (s == null) {
			ps.setNull(parameterIndex,Types.SMALLINT);
		}
		else {
			ps.setShort(parameterIndex,s.shortValue());
		}
	}

	/**
	 * Call <code>ps.setInt(parameterIndex,i.intValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.INTEGER)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param i An Integer object.
	 * @throws SQLException Upon errors.
	 */
	public static void setInt(PreparedStatement ps, int parameterIndex, Number i) throws SQLException {

		if (i == null) {
			ps.setNull(parameterIndex,Types.INTEGER);
		}
		else {
			ps.setInt(parameterIndex,i.intValue());
		}
	}

	/**
	 * Call <code>ps.setLong(parameterIndex,l.longValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.BIGINT)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param l A Long object.
	 * @throws SQLException Upon errors.
	 */
	public static void setLong(PreparedStatement ps, int parameterIndex, Number l) throws SQLException {

		if (l == null) {
			ps.setNull(parameterIndex,Types.BIGINT);
		}
		else {
			ps.setLong(parameterIndex,l.longValue());
		}
	}

	/**
	 * Call <code>ps.setFloat(parameterIndex,f.floatValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.FLOAT)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param f A Float object.
	 * @throws SQLException Upon errors.
	 */
	public static void setFloat(PreparedStatement ps, int parameterIndex, Number f) throws SQLException {

		if (f == null) {
			ps.setNull(parameterIndex,Types.FLOAT);
		}
		else {
			ps.setFloat(parameterIndex,f.floatValue());
		}
	}

	/**
	 * Call <code>ps.setDouble(parameterIndex,d.doubleValue())</code> or
	 * <code>ps.setNull(parameterIndex,Types.DOUBLE)</code> when <code>b</code>
	 * is <code>null</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                the first placeholder has the index 1.
	 * @param d A Double object.
	 * @throws SQLException Upon errors.
	 */
	public static void setDouble(PreparedStatement ps, int parameterIndex, Number d) throws SQLException {

		if (d == null) {
			ps.setNull(parameterIndex,Types.DOUBLE);
		}
		else {
			ps.setDouble(parameterIndex,d.doubleValue());
		}
	}

	/**
	 * Calls <code>ps.setString(parameterIndex,s)</code>.
	 *
	 * @param ps A prepared statement.
	 * @param parameterIndex The index of the placeholder in the statement to fill;
	 *                       the first placeholder has the index 1.
	 * @param s The string to set.
	 * @throws SQLException Upon errors.
	 */
	public static void setString(PreparedStatement ps, int parameterIndex, String s) throws SQLException {
		ps.setString(parameterIndex, s);
	}

	/** Helper function for the case, that utc seconds are stored in Java code in an attribute of type Double,
	 *  and mapped to a database column of type e.g. date, datetime or timestamp.  Transforms the given
	 *  utcSeconds into a java.sql.Date and calls the corresponding setter on the given statement.
     *  WARNING: The result depends on the time zone of the JVM.
	 *
	 * @param statement statement
	 * @param parameterIndex The index of the placeholder in the statement to fill;
     *                the first placeholder has the index 1.
	 * @param utcSeconds utc seconds as a Double object, can be null
	 * @throws SQLException upon errors
	 */
	public static void setUtcSeconds(PreparedStatement statement, int parameterIndex, Double utcSeconds) throws SQLException {
	    if (utcSeconds == null) {
	        statement.setNull(parameterIndex, Types.DATE);
	    } else {
	        long utcMillis = (long)(utcSeconds.doubleValue() * 1000);
	        Timestamp timeStamp = new Timestamp(utcMillis);
	        statement.setTimestamp(parameterIndex, timeStamp);
	    }
	}

	/** Set a parameter of SQL type TIMESTAMP to an Instant.
	 *  This function does not lose any precision.
	 * @param statement statement
	 * @param parameterIndex The index of the placeholder in the statement to fill;
     *                the first placeholder has the index 1.
	 * @param instant The instant, can be null.
	 * @throws SQLException upon errors
	 */
	public static void setInstant(PreparedStatement statement, int parameterIndex, Instant instant, ZoneId timeZone) throws SQLException {
	    if (instant == null) {
	        statement.setNull(parameterIndex, Types.TIMESTAMP);
	    } else {
	        statement.setTimestamp(parameterIndex, Timestamp.from(instant), timeZone == null ? null : Calendar.getInstance(TimeZone.getTimeZone(timeZone)));
	    }
	}

	public static void setInstant(PreparedStatement statement, int parameterIndex, Instant instant) throws SQLException {
		setInstant(statement, parameterIndex, instant, null);
	}

	public static void setI18nString(PreparedStatement statement, int parameterIndex, I18nString i18nString) throws SQLException {
	    StringBuilder s = new StringBuilder("{");

	    Set<String> languages = i18nString.getLanguages();
	    boolean first = true;
	    for (String language : languages) {
	        String translation = i18nString.getTranslation(language);

	        if (translation != null) {
	            if (!first) {
	                s.append(",");
	            }

	            s.append("\"");
	            s.append(language);
	            s.append("\":\"");
	            s.append(translation);
	            s.append("\"");

	            first = false;
	        }
	    }

	    s.append("}");

	    statement.setString(parameterIndex, s.toString());
	}

	/**
	 * Set an enum value as an integer parameter to a prepared statement
	 * using the ordinal of an enum value. If the enum value is <code>null</code>,
	 * <code>statement.setNull(pos,Types.INTEGER)</code> is performed.
	 *
	 * @param statement A prepared statement.
	 * @param pos The parameterIndex
	 * @param val The enum value to set.
	 * @throws SQLException Upon DB errors.
	 */
	public static <T extends Enum<T>> void setEnum(PreparedStatement statement, int pos, T val) throws SQLException {

		if (val == null) {
			statement.setNull(pos,Types.INTEGER);
		}
		else {
			statement.setInt(pos,val.ordinal());
		}
	}

	/**
	 * Fetch an enum value from an integer column of a result set interpreted
	 * as the ordinal of the given enum class.
	 *
	 * @param rs The result set to get the enum from.
	 * @param pos The index of the value to get.
	 * @param clazz The class of the requested enum value.
	 * @return An enum or <code>null</code>, if the column of the result set was null.
	 * @throws SQLException Upon DB errors.
	 */
	public static <T extends Enum<T>> T getEnum(ResultSet rs, int pos, Class<T> clazz) throws SQLException {

		int o = rs.getInt(pos);

		if (rs.wasNull()) {
			return null;
		}
		else {
			return clazz.getEnumConstants()[o];
		}
	}

}
