/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.ha;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import javax.sql.DataSource;

import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.jdbc.JdbcTransactionInterceptor;
import org.clazzes.util.sql.helper.JDBCTransaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A lock manager operating on a database table with the structure
 * <pre>
 *   SUBSYSTEM   VARCHAR(40) NOT NULL PRIMARY KEY,
 *   NODE_NAME   VARCHAR(40) NOT NULL,
 *   LOCK_TIME   BIGINT      NOT NULL,
 *   UNLOCK_TIME BIGINT      NOT NULL,
 *   STATUS      INT         NOT NULL
 * </pre>
 */
public class SqlHaLockManager implements IHaLockManager {

    private static final Logger log = LoggerFactory.getLogger(SqlHaLockManager.class); 
    
    private DataSource dataSource;
    private String connectionThreadLocalKey;
    private String nodeName;
    private String subSystem;
    private String tableName;
    
    public static final int STATUS_NO_LOCK =  2;
    public static final int STATUS_UNLOCKED = 0;
    public static final int STATUS_LOCKED   = 1;
    
    /**
     * Create a new lock manage instance.
     * 
     * Initialize this instance by calling
     * {@link #setDataSource(DataSource)}, {@link #setNodeName(String)},
     * {@link #setSubSystem(String)}, {@link #setTableName(String)} and
     * finally {@link #initialize()}.
     * 
     * If the table name is not configured, this instance is disabled
     * and return <code>null</code> locks in {@link #doLock()}
     */
    public SqlHaLockManager() {
    }
    
    /**
     * <p>
     * Initialize this lock manager after setting all configuration properties.
     * This function checks, whether there is a row in the lock table for the
     * given subsystem and creates such a row, is it is not present.</p>
     * 
     * <p>This function only works, if a datasource has been configured using
     * {@link #setDataSource(DataSource)}.</p>
     * 
     * @throws SQLException Upon DB errors.
     */
    public void initialize() throws SQLException {
        
        if (this.tableName == null) {
            
            log.info("Table name not configured, HA lock manager for subsystem [{}] is disabled.",
                    this.subSystem);
            return;
        }
        
        if (this.dataSource == null) {
            
            log.warn("No JDBC datasource configured, HA lock manager for subsystem [{}] will not be initialized.",
                    this.subSystem);
            return;
        }

       try (JDBCTransaction txn = new JDBCTransaction(this.dataSource,Connection.TRANSACTION_SERIALIZABLE)) {
            
            String sql = "select NODE_NAME,LOCK_TIME,UNLOCK_TIME,STATUS  from " + this.tableName + " where SUBSYSTEM=?";

            String oldNodeName = null;
            long oldLockTime = -1;
            long oldUnlockTime = -1;
            int oldStatus = -1;
            
            try (PreparedStatement ps = txn.getConnection().prepareStatement(sql)) {
                
                ps.setString(1,this.subSystem);
                
                try (ResultSet rs = ps.executeQuery()) {
                    
                    if (rs.next()) {
                        
                        oldNodeName = rs.getString(1);
                        oldLockTime = rs.getLong(2);
                        oldUnlockTime = rs.getLong(3);
                        oldStatus = rs.getInt(4);
                    }
                }
            }

            if (oldNodeName == null) {
                log.info("Table ["+this.tableName+"] does not contain a row for subsystem ["+this.subSystem+"], creating one now.");
                
                
                String sql2 = "insert into "+this.tableName+" (SUBSYSTEM,NODE_NAME,LOCK_TIME,UNLOCK_TIME,STATUS) values (?,?,?,?,?)";
                
                long now = System.currentTimeMillis();
                
                try (PreparedStatement ps = txn.getConnection().prepareStatement(sql2)) {
                    
                    ps.setString(1,this.subSystem);
                    ps.setString(2,this.nodeName);
                    ps.setLong(3,now);
                    ps.setLong(4,now);
                    ps.setInt(5,STATUS_NO_LOCK);
                    
                    ps.executeUpdate();
                }
                
            }
            else {
                log.info("Table ["+this.tableName+
                        "] contains a row for subsystem ["+this.subSystem+
                        "] with [NODE_NAME="+oldNodeName+
                        ",LOCK_TIME="+oldLockTime+
                        ",UNLOCK_TIME="+oldUnlockTime+
                        ",STATUS="+oldStatus+
                        "].");
            }
            txn.commit();
        }
    }
    
    private final class SqlHaLock implements IHaLock {

        private JDBCTransaction txn;
        private final String lastNodeName;
        private final long lastLockTime;
        private final long lastUnlockTime;
        private final long lockTime;
        
        
        public SqlHaLock(JDBCTransaction txn,
                long lockTime,
                String lastNodeName, long lastLockTime, long lastUnlockTime) {
            super();
            this.txn = txn;
            this.lastNodeName = lastNodeName;
            this.lastLockTime = lastLockTime;
            this.lastUnlockTime = lastUnlockTime;
            this.lockTime = lockTime;
        }

        @Override
        public void close() throws Exception {
            
            if (this.txn != null) {
                
                if (log.isDebugEnabled()) {
                    log.debug("Unlocking subsystem [{}] with node name [{}]",
                            SqlHaLockManager.this.subSystem,SqlHaLockManager.this.nodeName);
                }
                
                try {
                    SqlHaLockManager.this.updateStatus(this.txn,STATUS_UNLOCKED);
                    this.txn.commit();
                }
                catch (SQLException e) {
                    log.warn("Error propagating unlocked status for subsystem ["+
                            SqlHaLockManager.this.subSystem+
                            "] and node ["+
                            SqlHaLockManager.this.nodeName+"]");
                }
                
                JDBCTransaction tmp = this.txn;
                this.txn = null;
                tmp.close();
            }
        }

        @Override
        public long getLastLockTime() {
            
            return this.lastLockTime;
        }

        @Override
        public long getLastUnlockTime() {
            
            return this.lastUnlockTime;
        }

        @Override
        public long getLockTime() {
            
            return this.lockTime;
        }

        @Override
        public String getSubSystem() {
            
            return SqlHaLockManager.this.subSystem;
        }

        @Override
        public String getLastNodeName() {
          
            return this.lastNodeName;
        }

        @Override
        public String getNodeName() {
            
            return SqlHaLockManager.this.nodeName;
        }
    }
    
    protected long updateStatus(JDBCTransaction txn, int status) throws SQLException {
        
        long now = System.currentTimeMillis();
        
        String column = status == STATUS_UNLOCKED ? "UNLOCK_TIME" : "LOCK_TIME";
        
        String sql = "update " + this.tableName + " set NODE_NAME=?, "+column+"=?, STATUS=? where SUBSYSTEM=?";
        
        if (log.isDebugEnabled()) {
            log.debug("Updating ["+this.tableName+
                    "] for subsystem ["+this.subSystem+
                    "] with [NODE_NAME="+this.nodeName+
                    ","+column+"="+now+
                    ",STATUS="+status+
                    "]...");
        }

        try (PreparedStatement ps = txn.getConnection().prepareStatement(sql)) {
            
            ps.setString(1,this.nodeName);
            ps.setLong(2,now);
            ps.setInt(3,status);
            ps.setString(4,this.subSystem);
            
            ps.executeUpdate();
        }
        
        if (log.isDebugEnabled()) {
            log.debug("Successfully updated ["+this.tableName+
                    "] for subsystem ["+this.subSystem+
                    "] with [NODE_NAME="+this.nodeName+
                    ","+column+"="+now+
                    ",STATUS="+status+
                    "].");
        }

        return now;
    }
    
    protected SqlHaLock fetchAndLock(JDBCTransaction txn) throws SQLException {
        
        if (log.isDebugEnabled()) {
            log.debug("Locking subsystem [{}] with node name [{}]",this.subSystem,this.nodeName);
        }
        
        String sql = "select NODE_NAME,LOCK_TIME,UNLOCK_TIME,STATUS  from " + this.tableName + " where SUBSYSTEM=? for update";

        String oldNodeName;
        long oldLockTime;
        long oldUnlockTime;
        int oldStatus;
        
        try (PreparedStatement ps = txn.getConnection().prepareStatement(sql)) {
            
            ps.setString(1,this.subSystem);
            
            try (ResultSet rs = ps.executeQuery()) {
                
                if (rs.next()) {
                    
                    oldNodeName   = rs.getString(1);
                    oldLockTime   = rs.getLong(2);
                    oldUnlockTime = rs.getLong(3);
                    oldStatus     = rs.getInt(4);
                }
                else {
                    throw new SQLException("Table ["+this.tableName+"] does not contain a row for subsystem ["+this.subSystem+"].");
                }
            }
        }
        
        if (oldStatus == STATUS_LOCKED) {
            
            log.warn("Node [{}] has a stale lock on subsystem [{}] dating from [{}] to [{}]",
                    new Object[]{oldNodeName,this.subSystem,oldUnlockTime,oldLockTime});
        }
        else {
        
            if (log.isDebugEnabled()) {
                log.debug("Table ["+this.tableName+
                        "] contains a row for subsystem ["+this.subSystem+
                        "] with [NODE_NAME="+oldNodeName+
                        ",LOCK_TIME="+oldLockTime+
                        ",UNLOCK_TIME="+oldUnlockTime+
                        ",STATUS="+oldStatus+
                        "].");
            }
            
            if (oldStatus == STATUS_NO_LOCK) {
                oldNodeName = null;
            }
        }
        
        long lockTime = this.updateStatus(txn,STATUS_LOCKED);
        
        return new SqlHaLock(txn,lockTime,oldNodeName,oldLockTime,oldUnlockTime);
    }
    
    protected JDBCTransaction openTransaction() throws SQLException {
        
        if (this.connectionThreadLocalKey != null) {
            
            Connection conn = ThreadLocalManager.getBoundResource(this.connectionThreadLocalKey);
            
            if (conn == null) {
                throw new DAOException("No connection bound under key ["+this.connectionThreadLocalKey+
                        "], HA lock interceptor for subsystem ["+this.subSystem+
                        "] called outside of a transaction context.");
            }

            return new JDBCTransaction(conn);
        }
        else {
            return new JDBCTransaction(this.dataSource,Connection.TRANSACTION_READ_COMMITTED);
        }
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.util.sql.ha.IHaLockManger#doLock()
     */
    @Override
    public IHaLock doLock() throws SQLException {
        
        if (this.tableName == null) {
            
            if (log.isDebugEnabled()) {
                log.debug("HA lock manager for subsystem [{}] is disabled, not performing lock.",this.subSystem);
            }
            
            return null;
        }
        
        JDBCTransaction txn = openTransaction();
        
        try {
            
            SqlHaLock lock = this.fetchAndLock(txn);
            
            txn = null;
            return lock;
        }
        finally {
            if (txn != null) {
                try {
                    txn.close();
                }
                catch (SQLException e) {
                    log.warn("Error while closing transaction after failed attempt to lock subsystem ["+this.subSystem+"]",e);
                }
            }
        }
        
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.ha.IHaLockManager#getSubSystem()
     */
    public String getSubSystem() {
        return this.subSystem;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sql.ha.IHaLockManger#getNodeName()
     */
    @Override
    public String getNodeName() {
        
        return this.nodeName;
    }

    /**
     * Set the node name to report to the locking DB table.
     * If the supplied node name is <code>null</code> or empty,
     * the result of <code> InetAddress.getLocalHost().getHostName()</code>
     * is used or if this fails a random UUID is used.
     * 
     * @param nodeName The configured node name or <code>null</code> or
     *                 an empty string to use an auto-configured node name.
     */
    public void setNodeName(String nodeName) {
        
        if (nodeName == null || nodeName.isEmpty()) {
            try {
                this.nodeName = InetAddress.getLocalHost().getHostName();
            } catch (UnknownHostException e) {
                
                String uuidNodeName = UUID.randomUUID().toString();
                
                log.warn("Cannot determine local host name, using UUID ["+uuidNodeName+"] instead",e);
            }
        }
        else {
            this.nodeName = nodeName;
        }
    }
    
    /**
     * @param subSystem The subsystem used for locking.
     */
    public void setSubSystem(String subSystem) {
        
        this.subSystem = subSystem;
    }
    
    /**
     * Set the configured data source. This data source is only used by {@link #initialize()},
     * if a thread local key for the JDBC connection is provided via
     * {@link #setConnectionThreadLocalKey(String)}.
     * 
     * @param dataSource The SQL data source to use for this lock manager.
     */
    public void setDataSource (DataSource dataSource) {
        this.dataSource = dataSource;
    }

    /**
     * @return The configured JDBC data source.
     */
    public DataSource getDataSource() {
        return this.dataSource;
    }

    /**
     * @param connectionThreadLocalKey A thread local key to fetch database connections
     *     from. The database connection is presumably bound to the local thread
     *     by a {@link JdbcTransactionInterceptor} instance. If not set,
     *     create a connection from the provided data source.
     */
    public void setConnectionThreadLocalKey(String connectionThreadLocalKey) {
        this.connectionThreadLocalKey = connectionThreadLocalKey;
    }

    /**
     * @return The name of the thread local key to fetch database connections
     *     from.
     */
    public String getConnectionThreadLocalKey() {
        return this.connectionThreadLocalKey;
    }

    /**
     * @return The name of the database table to operate on.
     */
    public String getTableName() {
        return this.tableName;
    }

    /**
     * Set the name of the DB table used for locking.
     * The table has to have the following structure:
     * <pre>
     *   SUBSYSTEM   VARCHAR(40) NOT NULL PRIMARY KEY,
     *   NODE_NAME   VARCHAR(40) NOT NULL,
     *   LOCK_TIME   BIGINT      NOT NULL,
     *   UNLOCK_TIME BIGINT      NOT NULL,
     *   STATUS      INT         NOT NULL
     * </pre>
     * 
     * @param tableName The name of the database table to use. If
     *          set to <code>null</code>, this instance is disabled.
     */
    public void setTableName(String tableName) {
        this.tableName = tableName;
    }
    
}
