/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.dao;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

/**
 * A DAO interface for entities identified by IDs.
 *
 * @param <DTO> A data transfer object class.
 */
public interface IIdDAO<DTO> extends IBasicDAO<DTO> {

	/**
	 * @param dto A DTO.
	 * @return The ID of the DTO.
	 */
	Serializable getId(DTO dto);
	
	/**
	 * @return The class describing the IDs of the given DTO.
	 */
	Class<? extends Serializable> getIdClass();
	
    /**
     * @param id The ID of the DTO to retrieve.
     * @return The unique DTO with the given ID or <code>null</code>,
     *         if no DTO with the given ID exists.
     */
    public DTO get(Serializable id);
    
    /** 
     * Returns a List containing all DTOs for the given ids.
     * 
     * @param ids array of ids
     * @return List of corresponding DTOs, NOTE that the order in the list is in no way
     *         related to the order in the argument array.
     */
    public List<DTO> getBatch(final Serializable... ids);
    
    /** Returns a List containing all DTOs for the given ids.
     * 
     * @param ids List of ids
     * @return List of corresponding DTOs; NOTE that the order of the list is in no way
     *         related to the order of the argument list.
     */ 
    public List<DTO> getBatch(final Collection<? extends Serializable> ids);    
    
    /**
     * @param id The ID of the object to delete.
     * @return Whether the given ID was found in the database and the
     *         corresponding object has been deleted.
     */
    public boolean delete(Serializable id);
    
    /**
     * @param ids A collection of IDs of objects to delete.
     * @return An array of <code>ids.size()</code> counts
     *         of objects really deleted from the database,
     *         which contains 0 or 1 for each dto to be deleted.
     */
    public int[] deleteBatch(Collection<? extends Serializable> ids);
}
