/***********************************************************
 * $Id$
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql.dao;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.reflect.PropertyHelper;
import org.clazzes.util.sql.SQLDialect;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLConstruct;
import org.clazzes.util.sql.criteria.SQLValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>DAO framework class covering basic database-operations.</p>
 *
 * <p>This class implicitly support two ID policies:</p>
 * <ul>
 * <li>UUID-based, when the type of the ID property is String.</li>
 * <li>database-generated, when the type of the ID property is Long.</li>
 * </ul>
 *
 * <p>Database-generated keys work fine for mysql, but may have problems
 * with other database engines. For utmost database-independence, UUID-based
 * IDs are recommended.</p>
 *
 * <p>Since version 1.1 of sql-util you may provide for an ID generator for
 * {@link Long} IDs in order to generate unique IDs for all your entities and to
 * improve database-independence.</p>
 */
public abstract class AbstrIdDAO<T> extends AbstrBasicDAO<T> implements IIdDAO<T> {

	private static final Logger log = LoggerFactory.getLogger(AbstrIdDAO.class);

	private final String pkColumn;
	private final Class<T> clazz;
	private final Method idGetter;
	private final Method idSetter;
	private final int sqlTypeOfId;

	private IdGenerator idGenerator;

	/** Do not generate ids in save/saveBatch at all.  They are provided in the passed instances that are to be saved.
	 *  This mode is needed for weak entities.  E.g. B extends A, first save() an A-instance in a DAO with idGenerator
	 *  set, then copy the returned id into a B-instance, and save it in a DAO with idProvided == true.
	 *
	 */
	private boolean idProvided = false;

	/**
	 * Default Constructor.
	 *
	 * @see AbstrBasicDAO
	 *
	 * @param clazz The class object of <code>T</code> used to retrieved the getter for the id property.
	 * @param idProperty The name of the ID property. The according column may be named differently, it is specified as first column in {@code columns} and can be checked using {@link #getPkColumn()}.
	 * @param tableName Mandatory. The name of the table for which this DAO is used.
	 * @param columns Mandatory list of column names of the table for which this DAO is used. Is expected to begin with
	 * 			the primary key column which holds the id.
	 */
	public AbstrIdDAO(Class<T> clazz, String idProperty, String tableName, String... columns) {
		this(clazz, idProperty, tableName, columns, columns);
	}

	public AbstrIdDAO(Class<T> clazz, String idProperty, String tableName, String[] columns, String[] writableColumns) {
		super(tableName,columns,writableColumns);
		this.pkColumn = columns[0];
		this.clazz = clazz;
		try {

			this.idGetter = this.clazz.getMethod(PropertyHelper.getGetterName(idProperty));

			if (this.idGetter.getReturnType().equals(String.class)) {
				this.sqlTypeOfId = Types.VARCHAR;
			}
			else if (this.idGetter.getReturnType().equals(Long.class) ||
					this.idGetter.getReturnType().equals(long.class)) {
				this.sqlTypeOfId = Types.BIGINT;
			}
			else {
				 throw new IllegalArgumentException("ID property ["+idProperty+"] for bean class ["+clazz+"] is not of type java.lang.Long or java.lang.String");
			}

			this.idSetter =
					this.clazz.getMethod(PropertyHelper.getSetterName(idProperty),this.idGetter.getReturnType());

		} catch (NoSuchMethodException e) {
			 throw new IllegalArgumentException("Invalid ID property ["+idProperty+"] for bean class ["+clazz+"] specified",e);
		}
	}

	@Override
	public Serializable getId(T dto) {

		try {
			return (Serializable)this.idGetter.invoke(dto);
		} catch (IllegalAccessException e) {
			throw new IllegalArgumentException("Error getting ID property for bean class ["+this.clazz+"].",e);
		} catch (InvocationTargetException e) {
			throw new IllegalArgumentException("Error getting ID property for bean class ["+this.clazz+"].",e);
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public Class<? extends Serializable> getIdClass() {

		return (Class<? extends Serializable>)this.idGetter.getReturnType();
	}

	protected void setIdOnStatement(PreparedStatement statement, int pos, Object id) throws SQLException {

		if (AbstrIdDAO.this.sqlTypeOfId == Types.VARCHAR) {

			statement.setString(pos,(String)id);
		}
		else {

			statement.setLong(pos,((Number)id).longValue());
		}
	}

	/* (non-Javadoc)
	 * @see at.iteg.tis.graph.impl.dao.abstr.AbstrBasicDAO#save(java.lang.Object)
	 */
	@Override
	public T save(final T dto) {

        if (this.idProvided) {
            return super.save(dto);
        } else if (this.sqlTypeOfId == Types.VARCHAR) {
			try {
				this.idSetter.invoke(dto,this.generateUID());
			} catch (IllegalAccessException e) {
				throw new DAOException("Unable to set generated UUID to DTO of type ["+this.clazz+"]",e);
			} catch (InvocationTargetException e) {
				throw new DAOException("Unable to set generated UUID to DTO of type ["+this.clazz+"]",e);
			}

			return super.save(dto);
		} else if (this.idGenerator == null) {

			String sql = this.getGenerator().insert(
					this.getTableName(),
					SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()));

			if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

			if (this.sqlTypeOfId == Types.VARCHAR) {
				return performWithPreparedStatement(sql,
				new JdbcPreparedStatementAction<T>() {

					@Override
					public T perform(PreparedStatement statement) throws SQLException {

						fillPreparedStatementFromDto(statement, dto);

						statement.executeUpdate();

						return dto;
					}
				});
			}
			else {
				return performWithPreparedStatement(sql,
				new String[] {this.pkColumn},
				new JdbcPreparedStatementAction<T>() {

					@Override
					public T perform(PreparedStatement statement) throws SQLException {

						fillPreparedStatementFromDto(statement, dto);

						statement.execute();
						ResultSet rs = statement.getGeneratedKeys();
						rs.next();
						try {
							AbstrIdDAO.this.idSetter.invoke(dto,rs.getLong(1));
						} catch (IllegalAccessException e) {
							throw new DAOException("Unable to set generated ID to DTO of type ["+AbstrIdDAO.this.clazz+"]",e);
						} catch (InvocationTargetException e) {
							throw new DAOException("Unable to set generated ID to DTO of type ["+AbstrIdDAO.this.clazz+"]",e);
						}

						return dto;
					}
				});
			}
		} else {

			try {
				this.idSetter.invoke(dto,this.idGenerator.generateNext());
			} catch (IllegalAccessException e) {
				throw new DAOException("Unable to set generated ID to DTO of type ["+this.clazz+"]",e);
			} catch (InvocationTargetException e) {
				throw new DAOException("Unable to set generated ID to DTO of type ["+this.clazz+"]",e);
			}

			return super.save(dto);
		}
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sql.dao.AbstrBasicDAO#saveBatch(java.util.List)
	 */
	@Override
	public List<T> saveBatch(final List<T> dtos) {
		if (dtos.size() == 0) {
			return dtos;
		}

        if (this.idProvided) {
            return super.saveBatch(dtos);
        } else if (this.sqlTypeOfId == Types.VARCHAR) {

			try {

				for (T dto : dtos) {

					this.idSetter.invoke(dto,UUID.randomUUID().toString());
				}

			} catch (IllegalAccessException e) {
				throw new DAOException("Unable to set generated UUID to DTO of type ["+this.clazz+"]",e);
			} catch (InvocationTargetException e) {
				throw new DAOException("Unable to set generated UUID to DTO of type ["+this.clazz+"]",e);
			}

			return super.saveBatch(dtos);
		} else if (this.idGenerator == null) {

			String sql = this.getGenerator().insert(
					this.getTableName(),
					SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()));

			if (log.isDebugEnabled())
				log.debug("executing query [" + sql + "]");

			return performWithPreparedStatement(sql,new String[]{this.pkColumn},
							new JdbcPreparedStatementAction<List<T>>() {

				/* (non-Javadoc)
				 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
				 */
				@Override
				public List<T> perform(PreparedStatement statement) throws SQLException {

					for (T dto : dtos) {
						fillPreparedStatementFromDto(statement, dto);
						statement.addBatch();
					}

					statement.executeBatch();

					if (AbstrIdDAO.this.getGenerator().getDialect() == SQLDialect.MSSQL) {
						//
						// MSSQL JDBC Driver
						// https://search.maven.org/artifact/com.microsoft.sqlserver/mssql-jdbc
						// has limitation with generated keys for batch inserts, see
						//
						// https://github.com/microsoft/mssql-jdbc/issues/245
						// https://github.com/microsoft/mssql-jdbc/issues/358
						//
						log.warn("Unable to retrieve auto-generated IDs for MSSQL, see 'mssql-jdbc #245 #358' for details.");
					}
					else {
						ResultSet rs = statement.getGeneratedKeys();
						try {

							for (int i=0;i<dtos.size() && rs.next();++i) {
								T dto = dtos.get(i);

								if (log.isDebugEnabled()) {
									log.debug("Fetching auto-generated ID for [{}]",dto);
								}

								AbstrIdDAO.this.idSetter.invoke(dto,rs.getLong(1));

								if (log.isDebugEnabled()) {
									log.debug("Final DTO with auto-generated ID is [{}]",dto);
								}
							}

						} catch (IllegalAccessException e) {
							throw new DAOException("Unable to set generated ID to DTO of type ["+AbstrIdDAO.this.clazz+"]",e);
						} catch (InvocationTargetException e) {
							throw new DAOException("Unable to set generated ID to DTO of type ["+AbstrIdDAO.this.clazz+"]",e);
						}
					}
					return dtos;
				}
			});
		} else {
			try {
				for (T dto : dtos) {
					this.idSetter.invoke(dto,this.idGenerator.generateNext());
				}

			} catch (IllegalAccessException e) {
				throw new DAOException("Unable to set generated ID to DTO of type ["+this.clazz+"]",e);
			} catch (InvocationTargetException e) {
				throw new DAOException("Unable to set generated ID to DTO of type ["+this.clazz+"]",e);
			}

			return super.saveBatch(dtos);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see at.iteg.tis.graph.impl.dao.abstr.AbstrBasicDAO#update(java.lang.Object)
	 */
	@Override
	public int update(final T dto) {
		String sql = this.getGenerator().update(
				this.getTableName(),
				SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()),
				SQLCondition.eq(
						SQLValue.tableColumn(this.getTableName(), this.pkColumn),
						SQLValue.INSERT_VALUE));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<Integer>() {
			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public Integer perform(PreparedStatement statement) throws Exception {
				fillPreparedStatementFromDto(statement, dto);

				AbstrIdDAO.this.setIdOnStatement(statement,
						AbstrIdDAO.this.getWritableColumnNames().length+1,
						AbstrIdDAO.this.idGetter.invoke(dto));

				statement.execute();
				return statement.getUpdateCount();
			}
		});
	}

	@Override
	public int[] updateBatch(final Collection<T> dtos) {
		if (dtos.size() == 0)  {
			return new int[0];
		}

		String sql = this.getGenerator().update(
				this.getTableName(),
				SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()),
				SQLCondition.eq(
						SQLValue.tableColumn(this.getTableName(), this.pkColumn),
						SQLValue.INSERT_VALUE));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<int[]>() {
			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public int[] perform(PreparedStatement statement) throws Exception {

				for (T dto : dtos) {

					fillPreparedStatementFromDto(statement, dto);

					AbstrIdDAO.this.setIdOnStatement(statement,
							AbstrIdDAO.this.getWritableColumnNames().length+1,
							AbstrIdDAO.this.idGetter.invoke(dto));
					statement.addBatch();
				}

				return statement.executeBatch();
			}
		});
	}

	/**
	 * Get an Object for the given id or <code>null</code> if no such DTO exists.
	 *
	 * @param id The id for the queried DTO
	 * @return The matching Object or <code>null</code>
	 */
	public T get(final Serializable id) {

		SQLCondition cond = SQLCondition.eq(
						SQLValue.tableColumn(
								this.getTableName(),
								this.pkColumn),
								SQLValue.INSERT_VALUE);

		return super.getUniqueWithCondition(cond,new StatementPreparer() {

			@Override
			public void fillInsertValues(PreparedStatement statement)
					throws SQLException {
				AbstrIdDAO.this.setIdOnStatement(statement,1,id);
			}
		});
	}

	/**
     * Returns a List containing all DTOs for the given ids.
     *
     * @param ids array of ids
     * @return List of corresponding DTOs, NOTE that the order in the list is in no way
     *         related to the order in the argument array.
     */
    public List<T> getBatch(final Serializable... ids) {
        return this.getBatch(Arrays.asList(ids));
    }

	/** Returns a List containing all Objects for the given ids.
	 *
	 * @param ids List of ids
	 * @return List of corresponding objects; NOTE that the order of the list is in no way
	 *         related to the order of the argument list.
	 */
	public List<T> getBatch(final Collection<? extends Serializable> ids) {
		if (ids.size() == 0) {
			return new ArrayList<T>();
		}

	    SQLValue[] idValues = new SQLValue[ids.size()];
        int n = 0;
        for (Serializable id : ids) {
            idValues[n] = this.getSQLValueForId(id);
            n++;
        }

	    SQLCondition condition = SQLCondition.in(SQLValue.tableColumn(this.getTableName(), this.pkColumn),
	                                             SQLConstruct.commaParenthesis(idValues));

	    return super.getListWithCondition(condition, new StatementPreparer() {
	        public void fillInsertValues(PreparedStatement statement) throws SQLException {
	        }
	    });
	}

	protected SQLValue getSQLValueForId(Serializable id) {
        if (this.sqlTypeOfId != Types.BIGINT) {
            return SQLValue.stringValue(id.toString());
        } else {
            Long idAsLong = (Long)(id);
            return SQLValue.integerValue(idAsLong.longValue());
        }
	}

	/**
	 * Remove DTO for the given id.
	 */
	public boolean delete(final Serializable id) {
		String sql = this.getGenerator().delete(
				this.getTableName(),
				SQLCondition.eq(
						SQLValue.tableColumn(this.getTableName(), this.pkColumn),
						SQLValue.INSERT_VALUE));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<Boolean>() {
			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public Boolean perform(PreparedStatement statement) throws Exception {

				AbstrIdDAO.this.setIdOnStatement(statement,1,id);

				statement.execute();
				return true;
			}
		});
	}

	@Override
	public int[] deleteBatch(final Collection<? extends Serializable> ids) {
		if (ids.size() == 0) {
			return new int[0];
		}

		String sql = this.getGenerator().delete(
				this.getTableName(),
				SQLCondition.eq(
						SQLValue.tableColumn(this.getTableName(), this.pkColumn),
						SQLValue.INSERT_VALUE));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<int[]>() {
			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public int[] perform(PreparedStatement statement) throws Exception {

				for (Serializable id : ids) {

					AbstrIdDAO.this.setIdOnStatement(statement,1,id);
					statement.addBatch();
				}

				return statement.executeBatch();
			}
		});
	}

	/**
	 * Convenience method which generates a unique Id for this object. It implicitly checks with the primary key
	 * column to make sure that the id isn't already in use (the chance for this is estimated to be extremely low, but
	 * due to the fact that sufficient entropy can not be assumed, it is better to check).
	 *
	 * @return A String representing a UUID (see {@link UUID}).
	 */
	protected String generateUID() {
		String sql = this.getGenerator().select(
				this.getTableName(),
				SQLValue.valueList(
						SQLValue.tableColumn(this.getTableName(), this.pkColumn)),
						SQLCondition.eq(
								SQLValue.tableColumn(this.getTableName(), this.pkColumn),
								SQLValue.INSERT_VALUE));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<String>() {
			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public String perform(PreparedStatement statement) throws Exception {
				String uid = null;

				while (true) {
					uid = UUID.randomUUID().toString();

					statement.setString(1, uid);

					ResultSet rs = statement.executeQuery();

					if (rs.next()) {
						rs.close();
					} else
						break;
				}

				return uid;
			}
		});
	}

	/**
	 * @return The name of the primary key column.
	 */
	public String getPkColumn() {
		return this.pkColumn;
	}

	/**
	 * @return The class of the persisted entity type.
	 */
	public Class<T> getEntityClass() {
		return this.clazz;
	}

	/**
	 * @return An ID generator to use for IDs of type {@link Long}, which
	 *         disables the use of database-generated IDs.
	 */
	public IdGenerator getIdGenerator() {
		return this.idGenerator;
	}

	/**
	 * @param idGenerator  An ID generator to use for IDs
	 *         of type {@link Long}, which
	 *         disables the use of database-generated IDs.
	 */
	public void setIdGenerator(IdGenerator idGenerator) {
		this.idGenerator = idGenerator;

		if (idGenerator != null && this.sqlTypeOfId != Types.BIGINT) {
			log.warn("The provided ID generator will be ignored for IDs of type String, UUID-generation will be used instead.");
		}
	}

	public void setIdProvided(boolean idProvided) {
	    this.idProvided = idProvided;
	}
}
