/***********************************************************
 * $Id$
 *
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sql.dao;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.clazzes.util.aop.DAOException;
import org.clazzes.util.aop.jdbc.JdbcDAOSupport;
import org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction;
import org.clazzes.util.sql.SQLStatementGenerator;
import org.clazzes.util.sql.criteria.SQLCondition;
import org.clazzes.util.sql.criteria.SQLOrder;
import org.clazzes.util.sql.criteria.SQLValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A basic DAO support class for entities without a simple foreign key.
 *
 * @param <T> The DTO class.
 */
public abstract class AbstrBasicDAO<T> extends JdbcDAOSupport implements IBasicDAO<T> {

	private static final Logger log = LoggerFactory.getLogger(AbstrBasicDAO.class);

	private final String tableName;

	private final String[] columnNames;
	private final String[] writableColumnNames;
	private SQLStatementGenerator generator;


	public AbstrBasicDAO(String tableName, String... columnNames) {
		this(tableName, columnNames, columnNames);
	}

	public AbstrBasicDAO(String tableName, String[] columnNames,
			String[] writableColumnNames) {
		super();
		this.tableName = tableName;
		this.columnNames = columnNames;
		this.writableColumnNames = writableColumnNames;
	}

	public abstract int update(final T dto);

	@Override
	public int[] updateBatch(Collection<T> dtos) {
	    throw new UnsupportedOperationException();
	}

	public T save(final T dto) {
		String sql = this.getGenerator().insert(
				this.getTableName(),
				SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql,
				new JdbcPreparedStatementAction<T>() {

			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public T perform(PreparedStatement statement) throws SQLException {

				fillPreparedStatementFromDto(statement, dto);

				statement.executeUpdate();

				return dto;
			}
		});
	}

	public List<T> saveBatch(final List<T> dtos) {
		if (dtos.size() == 0) {
			return dtos;
		}

		String sql = this.getGenerator().insert(
				this.getTableName(),
				SQLValue.columnList(this.getTableName(), this.getWritableColumnNames()));

		if (log.isDebugEnabled())
			log.debug("executing query [" + sql + "]");

		return performWithPreparedStatement(sql,
				new JdbcPreparedStatementAction<List<T>>() {

			/* (non-Javadoc)
			 * @see org.clazzes.util.aop.jdbc.JdbcPreparedStatementAction#perform(java.sql.PreparedStatement)
			 */
			@Override
			public List<T> perform(PreparedStatement statement) throws SQLException {

				for (T dto : dtos) {
					fillPreparedStatementFromDto(statement, dto);
					statement.addBatch();
				}

				statement.executeBatch();
				return dtos;
			}
		});

	}

	/**
	 * Convenience method to fill in a POJO from a {@link ResultSet} which was obtained from the database.
	 * If the resultSet contains no items, the method must return <code>null</code>.
	 *
	 * @param rs The resultSet to parse
	 * @return A POJO of the correct type which was filled in from the resultSet
	 * @throws SQLException if the necessary parameters could not be obtained from the resultSet
	 */
	protected abstract T fillDtoFromResultSet(ResultSet rs)
			throws SQLException;

	/**
	 * Convenience method to set the values from a POJO to a {@link PreparedStatement}.
	 *
	 * @param statement The {@link PreparedStatement} to fill
	 * @param dto The POJO from which the values should be set
	 * @throws SQLException if the statement could not be filled as expected
	 */
	protected abstract void fillPreparedStatementFromDto(PreparedStatement statement,
			T dto) throws SQLException;

	/**
	 * Get's the sql that selects all the dto's that match the given condition and orders them according to orders.
	 * order will be unspecified if order is null or empty.
	 * @param condition The condition.
	 * @param orders The orderings.
	 * @return The sql statement.
	 */
	protected String getSelectSqlForCondition(SQLCondition condition, SQLOrder[] orders) {
		return this.getGenerator().select(this.getTableName(),
	            SQLValue.columnList(
	                    this.getTableName(),
	                    this.getColumnNames()),
	                    condition,
	                    orders);
	}

	/**
	 * Get's the sql that selects all the dto's for update
	 * that match the given condition and orders them according to orders.
	 * order will be unspecified if order is null or empty.
	 * @param condition The condition.
	 * @param orders The orderings.
	 * @return The sql statement.
	 */
	protected String getSelectForUpdateSqlForCondition(SQLCondition condition, SQLOrder[] orders) {
		return this.getGenerator().selectForUpdate(this.getTableName(),
	            SQLValue.columnList(
	                    this.getTableName(),
	                    this.getColumnNames()),
	                    condition,
	                    orders);
	}

	/**
	 * Get's the sql that selects all the dto's that match the given condition.
	 * The order is unspecified.
	 * @param condition The condition.
	 * @return The sql statement.
	 */
	protected String getSelectSqlForCondition(SQLCondition condition) {
		return this.getSelectSqlForCondition(condition, null);
	}

	/**
	 * Get's the sql that selects all the dto's for update
	 * that match the given condition.
	 * The order is unspecified.
	 * @param condition The condition.
	 * @return The sql statement.
	 */
	protected String getSelectForUpdateSqlForCondition(SQLCondition condition) {
		return this.getSelectSqlForCondition(condition, null);
	}

	/** Returns a list of POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
     * @param condition The condition for the <code>SELECT</code> query.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @return a List of POJOs as described
	 */
	protected List<T> getListWithCondition(SQLCondition condition, final StatementPreparer preparer) {
		return getSubList(condition, preparer, null, -1, -1);
	}

	/** Returns a list of POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
     * @param condition The condition for the <code>SELECT FOR UPDATE</code> query.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @return a List of POJOs as described
	 */
	protected List<T> getListWithConditionForUpdate(SQLCondition condition, final StatementPreparer preparer) {
		return getSubListForUpdate(condition, preparer, null, -1, -1);
	}

	/** Returns a list of POJOs, fetched using the given sql statement, prepared with the given StatementPreparer.
	 * @param sql an SQL statement returning the columns of the table matching DTO.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @return a List of POJOs as described
	 */
	protected List<T> getListWithSql(String sql, final StatementPreparer preparer) {
		return this.getSubListWithSql(sql, preparer, -1, -1);
	}

	/** Detects the number of POJOs that match would be returned by getList().
	 * @param condition an optional SQLCondition to be applied on a <code>SELECT * FROM foo WHERE condition</code>, where
	 *                  <code>foo</code> is the table matching DTO
	 * @param preparer an optional StatementPreparer to be applied on the condition
	 * @return number of POJOs as described
	 */
	protected long getListSize(SQLCondition condition, final StatementPreparer preparer) {

	    StringBuffer sql = new StringBuffer();
	    sql.append("SELECT COUNT(1) FROM ");
	    sql.append(this.getTableName());
	    if (condition != null) {
	        sql.append(" WHERE ");
	        sql.append(condition.toSQL(this.getGenerator().getDialect()));
	    }

	    return performWithPreparedStatement(sql.toString(), new JdbcPreparedStatementAction<Long>() {

	        public Long perform(PreparedStatement statement) throws Exception {

	            if (preparer != null) {
	                preparer.fillInsertValues(statement);
	            }

	            ResultSet rs = statement.executeQuery();

	            if (rs == null) {
	                return 0L;
	            }
	            if (!rs.next()) {
	                return 0L;
	            }
	            return rs.getLong(1);
	        }
	    });
	}

	/** Returns a list of POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
	 * @param condition an SQLCondition to be applied on a <code>SELECT * FROM foo WHERE condition</code>, where
	 *                  <code>foo</code> is the table matching DTO.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @param orders an optional array of {@link SQLOrder} objects to apply, use null to suppress ordering
	 * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
	 * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
	 * @return a List of POJOs as described
	 */
	protected List<T> getSubList(SQLCondition condition, final StatementPreparer preparer, final SQLOrder[] orders, final int rowsToSkip, final int maxRows) {
		return this.getSubListWithSql(this.getSelectSqlForCondition(condition, orders), preparer, rowsToSkip, maxRows);
	}

	/** Returns a list of POJOs for update,
	 * matching the given SQLCondition, prepared with the given StatementPreparer.
	 * @param condition an SQLCondition to be applied on a <code>SELECT FOR UPDATE * FROM foo WHERE condition</code>, where
	 *                  <code>foo</code> is the table matching DTO.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @param orders an optional array of {@link SQLOrder} objects to apply, use null to suppress ordering
	 * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
	 * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
	 * @return a List of POJOs as described
	 */
	protected List<T> getSubListForUpdate(SQLCondition condition, final StatementPreparer preparer, final SQLOrder[] orders, final int rowsToSkip, final int maxRows) {
		return this.getSubListWithSql(this.getSelectForUpdateSqlForCondition(condition, orders), preparer, rowsToSkip, maxRows);
	}

	/** Returns a list of POJOs, fatched using the given sql statement, prepared with the given StatementPreparer.
     * @param sql The sql statement, must return the same columns as this dto.
	 * @param preparer a StatementPreparer to be applied on the condition
	 * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
	 * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
	 * @return a List of POJOs as described
	 */
	protected List<T> getSubListWithSql(String sql, final StatementPreparer preparer, final int rowsToSkip, final int maxRows) {

	    List<T> list = new ArrayList<T>(Math.max(maxRows,10));

        IEntityConsumer<T> consumer = new AppendListEntityConsumer<T>(list);

        this.streamSubListWithSql(consumer, sql, preparer, rowsToSkip, maxRows);

        return list;
	}

    /** Stream a list of POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
     * @param entityConsumer The consumer for the received entites.
     * @param condition an SQLCondition to be applied on a <code>SELECT * FROM foo WHERE condition</code>, where
     *                  <code>foo</code> is the table matching DTO
     * @param preparer a StatementPreparer to be applied on the condition
     * @return a List of POJOs as described
     */
    protected int streamListWithCondition(
            final IEntityConsumer<T> entityConsumer,
            SQLCondition condition, final StatementPreparer preparer) {
      return streamSubList(entityConsumer, condition, preparer, null, -1, -1);
    }

    protected int streamListWithSql(
            final IEntityConsumer<T> entityConsumer,
            String sql, final StatementPreparer preparer) {
      return streamSubListWithSql(entityConsumer, sql, preparer, -1, -1);
    }

    /** Stream a list of POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
     * @param entityConsumer The consumer for the received entites.
     * @param condition an SQLCondition to be applied on a <code>SELECT * FROM foo WHERE condition</code>, where
     *                  <code>foo</code> is the table matching DTO.
     * @param preparer a StatementPreparer to be applied on the condition
     * @param orders an optional array of {@link SQLOrder} objects to apply, use null to suppress ordering
     * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
     * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
     * @return The number of POJOs stream to <code>entityConsumer</code>.
     */
	protected int streamSubList(
	        final IEntityConsumer<T> entityConsumer,
	        SQLCondition condition, final StatementPreparer preparer, final SQLOrder[] orders, final int rowsToSkip, final int maxRows) {
		return this.streamSubListWithSql(entityConsumer, this.getSelectSqlForCondition(condition, orders), preparer, rowsToSkip, maxRows);
	}

    /** Stream a list of POJOs for update,
	 * matching the given SQLCondition, prepared with the given StatementPreparer.
     * @param entityConsumer The consumer for the received entites.
     * @param condition an SQLCondition to be applied on a <code>SELECT FOR UPDATE * FROM foo WHERE condition</code>, where
     *                  <code>foo</code> is the table matching DTO.
     * @param preparer a StatementPreparer to be applied on the condition
     * @param orders an optional array of {@link SQLOrder} objects to apply, use null to suppress ordering
     * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
     * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
     * @return The number of POJOs stream to <code>entityConsumer</code>.
     */
	protected int streamSubListForUpdate(
	        final IEntityConsumer<T> entityConsumer,
	        SQLCondition condition, final StatementPreparer preparer, final SQLOrder[] orders, final int rowsToSkip, final int maxRows) {
		return this.streamSubListWithSql(entityConsumer, this.getSelectForUpdateSqlForCondition(condition, orders), preparer, rowsToSkip, maxRows);
	}

	/** Stream a list of POJOs, fetched using the given sql statement, prepared with the given StatementPreparer.
     * @param entityConsumer The consumer for the received entites.
     * @param sql The sql statement, must return the same columns as this dto.
     * @param preparer a StatementPreparer to be applied on the condition
     * @param rowsToSkip the rows to skip or a value &le;0 to not skip anything. Think of the OFFSET sub option of MySQL's LIMIT option.
     * @param maxRows    the maximum number of rows to return, or -1 for no limit. Think of MySQL's LIMIT option.
     * @return The number of POJOs stream to <code>entityConsumer</code>.
     */
	protected int streamSubListWithSql(
	        final IEntityConsumer<T> entityConsumer,
	        String sql, final StatementPreparer preparer, final int rowsToSkip, final int maxRows) {

	    return this.performWithPreparedStatement(sql,
	            rowsToSkip > 0 ? ResultSet.TYPE_SCROLL_INSENSITIVE : ResultSet.TYPE_FORWARD_ONLY,
	                    ResultSet.CONCUR_READ_ONLY,
	                    new JdbcPreparedStatementAction<Integer>() {

	                public Integer perform(PreparedStatement statement) throws Exception {

	                    if (preparer != null) {
	                        preparer.fillInsertValues(statement);
	                    }

	                    if (maxRows >= 0) {
	                        statement.setFetchSize(maxRows == 0 ? 1 : maxRows);
	                        if (rowsToSkip > 0) {
	                            statement.setMaxRows(rowsToSkip + maxRows);
	                        }
	                    }
	                    ResultSet rs = statement.executeQuery();

	                    if (rowsToSkip > 0) {

	                        rs.absolute(rowsToSkip);
	                    }

	                    int nEntities = 0;

	                    while (rs.next() && (maxRows < 0 || nEntities < maxRows)) {

                            entityConsumer.consumeEntity(fillDtoFromResultSet(rs));
                            ++nEntities;
	                    }

	                    return nEntities;
	                }
	            });
	}


	/** Returns a POJO, fetched using the given sql statement, prepared with the given StatementPreparer.
     *  The sql statement is expected to fetch at most one tuple on the database.  If none is found,
     *  null is returned, if one is found, the corresponding POJO is returned, if more than one are
     *  found, a DAOException is thrown.
     * @param sql The sql statement, must return the same columns as this dto.
     * @param preparer a StatementPreparer to be applied on the condition
     * @return a POJO as described
     */
	protected T getUniqueWithSql(String sql, final StatementPreparer preparer) {
		return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<T>() {

			public T perform(PreparedStatement statement) throws Exception {

				if (preparer != null) {
					preparer.fillInsertValues(statement);
				}

				ResultSet rs = statement.executeQuery();
				T ret = null;

				while (rs.next()) {

					if (ret != null) {

						throw new DAOException("getUniqueWithSql found multiple "
								+ " instances of [" + AbstrBasicDAO.this.getTableName() + "] where at most one was expected.");
					}
					else {
						ret = fillDtoFromResultSet(rs);
					}
				}
				return ret;
			}
		});

	}

	/** Returns a POJO, matching the given SQLCondition, prepared with the given StatementPreparer.
     *  The condition is expected to meet for at most one tuple on the database.  If none is found,
     *  null is returned, if one is found, the corresponding POJO is returned, if more than one are
     *  found, a DAOException is thrown.
     * @param condition an SQLCondition to be applied on a SELECT * FROM foo WHERE condition, where
     *                  foo is the table matching DTO
     * @param preparer a StatementPreparer to be applied on the condition
     * @return a POJO as described
     */
	protected T getUniqueWithCondition(SQLCondition condition, final StatementPreparer preparer) {

		String sql = this.getSelectSqlForCondition(condition);

		return this.getUniqueWithSql(sql, preparer);
	}

	/** Returns a POJO for update,
	 *  matching the given SQLCondition, prepared with the given StatementPreparer.
     *  The condition is expected to meet for at most one tuple on the database.  If none is found,
     *  null is returned, if one is found, the corresponding POJO is returned, if more than one are
     *  found, a DAOException is thrown.
     * @param condition an SQLCondition to be applied on a SELECT * FROM foo WHERE condition, where
     *                  foo is the table matching DTO
     * @param preparer a StatementPreparer to be applied on the condition
     * @return a POJO as described
     */
	protected T getUniqueWithConditionForUpdate(SQLCondition condition, final StatementPreparer preparer) {

		String sql = this.getSelectForUpdateSqlForCondition(condition);

		return this.getUniqueWithSql(sql, preparer);
	}

	/** Delete all POJOs, matching the given SQLCondition, prepared with the given StatementPreparer.
     *
     * @param condition an SQLCondition to be applied on a <code>DELETE FROM foo WHERE condition</code>,
     *                  where <code>foo</code> is the table matching DTO.
     * @param preparer a StatementPreparer to be applied on the condition
     * @return The number of POJOs deleted.
     */
	protected int deleteWithCondition(SQLCondition condition, final StatementPreparer preparer) {

	    String sql = this.getGenerator().delete(this.getTableName(),condition);

	    return performWithPreparedStatement(sql, new JdbcPreparedStatementAction<Integer>() {

	        public Integer perform(PreparedStatement statement) throws Exception {

	            if (preparer != null) {
	                preparer.fillInsertValues(statement);
	            }

	            return statement.executeUpdate();
	        }
       });
	}

	/** Returns POJOs for all tuples in this.getTableName().
	 *
	 * @return POJOs for all tuples in this.getTableName()
	 */
	@Override
	public List<T> getAll() {
	    return getSubList(null, null, null, -1, -1);
	}

	/** Returns POJOs for update of all tuples in this.getTableName().
	 *
	 * @return POJOs for all tuples in this.getTableName()
	 */
	@Override
	public List<T> getAllForUpdate() {
	    return getSubListForUpdate(null, null, null, -1, -1);
	}

	/**
	 * @param generator the generator to set
	 */
	public void setGenerator(SQLStatementGenerator generator) {
		this.generator = generator;
	}

	/**
	 * @return the generator used by this instance.
	 */
	public SQLStatementGenerator getGenerator() {
		return this.generator;
	}

	/**
	 * @return The name of the SQL table.
	 */
	public String getTableName() {
		return this.tableName;
	}

	/**
	 * @return The list of table column names. The first column is the primary key.
	 */
	public String[] getColumnNames() {
		return this.columnNames;
	}

	/**
	 * @return The list of table column names. The first column is the primary key.
	 */
	public String[] getWritableColumnNames() {
		return this.writableColumnNames;
	}

	/**
	 * Get the table column name with in dex <code>i</code>.
	 *
	 * @param i The index.
	 * @return The corresponding column name.
	 */
	public String getColumnName(int i) {
		return this.columnNames[i];
	}

	/**
	 * @return The number of database columns.
	 */
	public int getColumnCount() {
		return this.columnNames.length;
	}

}
