/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql.criteria;

import org.apache.commons.lang.StringEscapeUtils;
import org.clazzes.util.sql.SQLDialect;
import org.clazzes.util.sql.SQLFragment;

/**
 * A convenience class for wrapping column information and placeholder values for SQL statement generation
 * 
 * @author jpayr
 *
 */
public class SQLValue implements SQLFragment {
	
	/**
	 * An internal Enumeration to denote the types of values a {@link SQLValue} can represent
	 * 
	 * @author jpayr
	 *
	 */
	public enum ValueType {
		TABLE_COLUMN,
		PLACEHOLDER,
		INTEGER_VALUE,
		STRING_VALUE
	}
	
	
	private static final long serialVersionUID = -7768523896059257899L;

	public static final SQLValue INSERT_VALUE = new SQLValue(ValueType.PLACEHOLDER, "?");
	
	private String[] values;
	
	private ValueType type;
	
	protected SQLValue(ValueType type, String... values) {
		super();
		this.type = type;
		this.values = values;
	}
	
	/**
	 *  Creates a {@link SQLValue} representing a long constant.
	 * 
	 * @param longValue A long value to use as SQL literal.
	 * @return An {@link SQLValue} for the given integer literal.
	 */
	public static SQLValue integerValue(long longValue) {
	    return new SQLValue(ValueType.INTEGER_VALUE, Long.toString(longValue));
	}
	
	/**
	 *  Creates a {@link SQLValue} representing a string constant.
	 *  
	 * @param value A string to use as SQL literal.
	 * @return An {@link SQLValue} for the given string literal.
	 */
	public static SQLValue stringValue(String value) {
	    return new SQLValue(ValueType.STRING_VALUE, value);
	}
	
	/**
	 * Create a {@link SQLValue} representation of an arbitrary table column
	 * 
	 * @param tableName
	 * @param columnName
	 * @return An identifier representing the column of the given table.
	 */
	public static SQLValue tableColumn(String tableName, String columnName) {
		return new SQLValue(ValueType.TABLE_COLUMN, tableName, columnName);
	}
	
	/**
	 * Convenience method to create a list of {@link SQLValue}s
	 * @param values
	 * @return An identifier representing a list of values.
	 */
	public static SQLValue[] valueList(SQLValue... values) {
		return values;
	}
	
	/**
	 * Convenience method to create a list of {@link SQLValue}s representing columns from one table
	 * @param tableName
	 * @param columnNames
	 * @return An identifier representing a list of column of the given table.
	 */
	public static SQLValue[] columnList(String tableName, String... columnNames) {
		SQLValue[] results = new SQLValue[columnNames.length];
		for (int i=0; i < columnNames.length; i++) {
			results[i] = new SQLValue(ValueType.TABLE_COLUMN, tableName, columnNames[i]);
		}
		return results;
	}
	
	/* (non-Javadoc)
	 * @see org.clazzes.topicmaps.jdbc.sql.criteria.SQLFragment#toSQL(org.clazzes.topicmaps.jdbc.sql.SQLDialect)
	 */
	@Override
	public String toSQL(SQLDialect dialect) {
		switch (this.type) {
		case TABLE_COLUMN:
			switch (dialect) {
			default:
				return this.values[0] + "." + this.values[1];
			}
			
		case PLACEHOLDER:
        case INTEGER_VALUE:
			switch (dialect) {
			default:
				return this.values[0];
			}
		    
		case STRING_VALUE:
		    switch (dialect) {
		    default:
		        return "'" + StringEscapeUtils.escapeSql(this.values[0]) + "'";
		    }

		default:
			break;
		}
		
		return null;
	}

    /**
     * @param dialect The SQL dialect to choose
     * @return An unqualified SQL iddentifier, for table columns, this omits the table name before the dot.
     */
    public String toUnqualifiedSQL(SQLDialect dialect) {
        
        if (this.type == ValueType.TABLE_COLUMN) {
            return this.values[1];    
        }
        else {
            return this.toSQL(dialect);
        }
    }
}
