/***********************************************************
 * $Id$
 * 
 * SQL/DAO utilities of clazzes.org
 * http://www.clazzes.org
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sql;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A factory to deduce an {@link SQLDialect} from a database product name.
 */
public abstract class SQLDialectFactory {

    private static final Logger log = LoggerFactory
            .getLogger(SQLDialectFactory.class);
    
    /**
     * @param databaseProduct A database product name as returned by
     *               {@link DatabaseMetaData#getDatabaseProductName()}
     * @return The matching {@link SQLDialect} for the DML queries or
     *            {@link SQLDialect#ISO}, if the product is unknown.
     */
    public static final SQLDialect getSQLDialect(String databaseProduct) {
        
        SQLDialect ret = SQLDialect.ISO;
        
        if (databaseProduct.contains("MariaDB") || databaseProduct.contains("MySQL")) {
            ret = SQLDialect.MYSQL;
        }
        else if (databaseProduct.contains("Oracle")) {
            ret = SQLDialect.ORACLE;
        }
        else if (databaseProduct.contains("Microsoft SQL Server")) {
            ret = SQLDialect.MSSQL;
        }
        else if (databaseProduct.contains("PostgreSQL")) {
            ret = SQLDialect.POSTGRES;
        }
        else if (databaseProduct.contains("Apache Derby")) {
            ret = SQLDialect.DERBY;
        }
        else {
            log.warn("DB product [{}] is unknown, will use [{}] as dialect.",
                    databaseProduct,ret);
        }
        
        log.info("Using DML dialect [{}] for DB product [{}].",
                ret,databaseProduct);
        
        return ret;
    }
    
    /**
     * @param metaData The database meta data to deduce the DML dialect from.
     * @return The matching {@link SQLDialect} for the DML queries or
     *            {@link SQLDialect#ISO}, if the product is unknown.
     * @throws SQLException Upon connection errors.
     */
    public static final SQLDialect getSQLDialect(DatabaseMetaData metaData) throws SQLException {
        
        return getSQLDialect(metaData.getDatabaseProductName());
    }

    /**
     * @param connection The database connection to deduce the DML dialect from.
     * @return The matching {@link SQLDialect} for the DML queries or
     *            {@link SQLDialect#ISO}, if the product is unknown.
     * @throws SQLException Upon connection errors.
     */
    public static final SQLDialect getSQLDialect(Connection connection) throws SQLException {
        
        return getSQLDialect(connection.getMetaData());
    }

    /**
     * @param dataSource The data source (connection pool) to deduce the DML dialect from.
     * @return The matching {@link SQLDialect} for the DML queries or
     *            {@link SQLDialect#ISO}, if the product is unknown.
     * @throws SQLException Upon connection errors.
     */
    public static final SQLDialect getSQLDialect(DataSource dataSource) throws SQLException {
        
        try (Connection connection = dataSource.getConnection()) {
            return getSQLDialect(connection);
        }
    }
}
