/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 16.02.2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.PriorityQueue;

import org.clazzes.util.aop.IFileDeleter;
import org.clazzes.util.sched.cache.IScratchBucket;
import org.clazzes.util.sched.cache.IScratchFileCache;
import org.clazzes.util.sched.cache.ScratchFile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A cache, which caches temporary files up to a configured watermark.
 * 
 * This cache is designed in order to keep files as long as the JVM is running.
 * 
 */
public class ScratchFileCacheImpl implements IScratchFileCache {

    private static final Logger log = LoggerFactory.getLogger(ScratchFileCacheImpl.class);
    
    private static final double MB_FAC = 1.0 / 1024.0 / 1024.0;
    
    private double cacheSizeMB;
    private long maxWaitMillis;
    private IFileDeleter fileDeleter;
    
    private double currentCacheSizeMB;

    private static final class EntryLock {

        private boolean finished;
        
        public synchronized void waitForFinish(long timeout) throws InterruptedException {
            
            if (!this.finished) {
               this.wait(timeout); 
            }
        }
        
        public synchronized void finish() {
            this.finished = true;
            this.notifyAll();
        }
    }
    
    private static final class CacheEntry implements Comparable<CacheEntry> {
        
        private final String key;
        private final ScratchFile scratchFile;
        private final double entryMB;
        private long lastAccess;
        
        public CacheEntry(String key, ScratchFile scratchFile, double entryMB) {
            
            this.key = key;
            this.scratchFile = scratchFile;
            this.entryMB = entryMB;
            this.touch();
        }

        public String getKey() {
            return this.key;
        }

        public ScratchFile getScratchFile() {
            return this.scratchFile;
        }

        public double getEntryMB() {
            return this.entryMB;
        }

        public long getLastAccess() {
            return this.lastAccess;
        }

        public void touch() {
            this.lastAccess = System.currentTimeMillis();
        }

        @Override
        public int compareTo(CacheEntry o) {
            
            if (this.lastAccess < o.lastAccess) {
                return 1;
            }
            else if (this.lastAccess > o.lastAccess) {
                return -1;
            }
            else {
                return 0;
            }
        }
    }
    
    private class ScratchBucketImpl implements IScratchBucket {

        private final String key; 
        private ScratchFile scratchFile;
        private boolean closeScratchFile;
        private boolean closed;
        
        public ScratchBucketImpl(String key, ScratchFile scratchFile) {
            
            this.key = key;
            this.scratchFile = scratchFile;
        }
        
        @Override
        public void close() throws IOException {
            
            if (!this.closed) {
                
                this.closed = true;
                
                if (this.scratchFile == null) {
                    ScratchFileCacheImpl.this.cancelAnnouncement(this.key);
                }
                else if (this.closeScratchFile) {
                    
                    this.scratchFile.deleteAndClose();
                }
            }
        }

        @Override
        public String getKey() {
            
            return this.key;
        }

        @Override
        public ScratchFile getScratchFile() {
            
            return this.scratchFile;
        }

        @Override
        public ScratchFile provideScratchFile(File file, String mimeType,
                String disposition) {
            
            this.scratchFile = ScratchFileCacheImpl.this.newScratchFile(file, mimeType, disposition);
            
            this.closeScratchFile = !ScratchFileCacheImpl.this.addScratchFile(this.key,this.scratchFile);
            
            return this.scratchFile;
        }
        
    }
    
    private final Map<String,CacheEntry> entriesByKey;
    private final Map<String,EntryLock> announcedKeys;
    
    /**
     * Create a new scratch file cache instance. Please call
     * {@link #destroy()}, when this instance will not be used anymore.
     */
    public ScratchFileCacheImpl() {
        this.entriesByKey = new HashMap<String,CacheEntry>();
        this.announcedKeys = new HashMap<String,EntryLock>();
    }

    /**
     * Create a new scratch file using the specified file deleter.
     * 
     * @param file The file to be cached
     * @param mimeType The MIME type of the content
     * @param disposition The content-disposition a known from the likewise HTTP header.
     * @return a new scratch file instance.
     */
    public ScratchFile newScratchFile(File file, String mimeType, String disposition) {
    	
    	return new ScratchFile(this.fileDeleter,file,mimeType,disposition);
    }
    
    /**
     * Provide new content under the given application-specific cache key.
     * 
     * @param key The application-specific key under which to proved a new scratch file.
     * @param scratchFile The data to provide.
     * @return Whether the file has been cached. If <code>false</code> is returned, the
     *         size of the file is greater than a tenth of the overall cache size.
     */
    public synchronized boolean addScratchFile(String key, ScratchFile scratchFile) {
        
       double entryMB = scratchFile.getFile().length() * MB_FAC;
       
       Object lock = this.announcedKeys.remove(key);
       
       if (lock != null) {
           
           synchronized (lock) {
               lock.notifyAll();
           }
       }

       if (entryMB > this.cacheSizeMB * 0.1) {
           
           if (log.isDebugEnabled()) {
               log.debug("Rejecting too large entry [{}] for key [{}] with size [{}MB].",
                       new Object[] {scratchFile,key,entryMB});
           }

           return false;
       }
       
       if (log.isDebugEnabled()) {
           log.debug("Adding entry [{}] for key [{}] with size [{}MB].",
                   new Object[] {scratchFile,key,entryMB});
       }

       CacheEntry oldEntry = this.entriesByKey.put(key,new CacheEntry(key,scratchFile,entryMB));
       
       if (oldEntry != null) {
           
           log.warn("Purging duplicate entry [{}] for key [{}].",oldEntry.getScratchFile(),key);
           
           oldEntry.getScratchFile().deleteAndClose();
           this.currentCacheSizeMB -= oldEntry.getEntryMB();
       }
       
       this.currentCacheSizeMB += entryMB;
       
       if (log.isDebugEnabled()) {
           log.debug("Total cache size is now [{}MB].",this.currentCacheSizeMB);
       }
        
       if (this.currentCacheSizeMB > this.cacheSizeMB) {
           
           PriorityQueue<CacheEntry> entriesToBePurged = new PriorityQueue<ScratchFileCacheImpl.CacheEntry>(256);
           
           double toPurgeMB = this.currentCacheSizeMB - this.cacheSizeMB * 0.9; 
           double purgedMB = 0.0;
           
           
           for (Map.Entry<String,CacheEntry> e : this.entriesByKey.entrySet()) {
               
               if (e.getKey().equals(key)) {
                   continue;
               }
               
               entriesToBePurged.add(e.getValue());
               purgedMB += e.getValue().getEntryMB();
               
               // take the youngest entry to be purged and assure, that
               // at least one entry remains to be purged.
               while (purgedMB > toPurgeMB && entriesToBePurged.size() > 1) {
                   
                   CacheEntry leftEntry = entriesToBePurged.poll();
                   purgedMB -= leftEntry.getEntryMB();
               }
           }
           
           for (CacheEntry e : entriesToBePurged) {

               if (log.isDebugEnabled()) {
                   log.debug("Garbage collecting entry [{}] for key [{}], last modified at [{}].",
                           new Object[]{e.getScratchFile(),e.getKey(),e.getLastAccess()});
               }

               this.entriesByKey.remove(e.getKey());
               this.currentCacheSizeMB -= e.getEntryMB();
           }
       }
       
       return true;
    }
    
    /**
     * Cancel an announcement placed by {@link #getScratchFile(String)} in the 
     * case a <code>null</code> scratch file is returned.
     * 
     * @param key The application-specific key of the announcement to cancel.
     */
    public synchronized void cancelAnnouncement(String key) {
        
        EntryLock lock = this.announcedKeys.remove(key);
        
        if (lock != null) {
            
            if (log.isDebugEnabled()) {
                log.debug("Cancelling announcement for key [{}].",key);
            }

            lock.finish();
        }
    }
    
    private synchronized Object getScratchFileOrLock(String key) {
        
        CacheEntry entry = this.entriesByKey.get(key);
        
        if (entry == null) {
            
            if (log.isDebugEnabled()) {
                log.debug("Found no entry for key [{}].",key);
            }
 
            EntryLock lock = this.announcedKeys.get(key);
            
            if (lock == null) {
                
                if (log.isDebugEnabled()) {
                    log.debug("Announcing creation of entry for key [{}].",key);
                }

                this.announcedKeys.put(key,new EntryLock());

                return null;
            }
            else {
                
                return lock;
            }
        }
        else {
            entry.touch();
            return entry.getScratchFile();
        }
    }
    
    /**
     * Check for data under the given key. If another thread has announced the
     * creation of the resource, this method waits up to
     * {@link #getMaxWaitMillis()} milliseconds for the other thread to provide
     * the resource under the given key.
     * 
     * @param key The application-specific key to check for cached data.
     * @return A cached scratch file or <code>null</code>, which means that
     *         an announcement is created, that should either be confirmed with
     *         {@link #addScratchFile(String, ScratchFile)} or cancelled with
     *         {@link #cancelAnnouncement(String)} by the calling thread.
     * @throws InterruptedException If the wait 
     */
    public ScratchFile getScratchFile(String key) throws InterruptedException {
        
        Object scratchFileOrLock = this.getScratchFileOrLock(key);
        
        if (scratchFileOrLock == null) {
            
            return null;
        }
        else if (scratchFileOrLock.getClass() == EntryLock.class) {
            
            if (log.isDebugEnabled()) {
                log.debug("Waiting [{}ms] for another thread to provide data for key [{}]...",this.maxWaitMillis,key);
            }

            EntryLock lock = (EntryLock) scratchFileOrLock;
            
            lock.waitForFinish(this.maxWaitMillis);
            
            synchronized (this) {
                CacheEntry entry = this.entriesByKey.get(key);
                
                if (entry == null) {
                    
                    if (log.isWarnEnabled()) {
                        log.warn("No data after wait for [{}ms] for another thread to provide data for key [{}].",this.maxWaitMillis,key);
                    }

                    return null;
                }
                else {
                    
                    if (log.isDebugEnabled()) {
                        log.debug("Another thread provided entry [{}] for key [{}].",entry.getScratchFile(),key);
                    }
                    
                    entry.touch();
                    return entry.getScratchFile();
                }
            }

        }
        else {
            
            if (log.isDebugEnabled()) {
                log.debug("Found existing entry [{}] for key [{}].",scratchFileOrLock,key);
            }

            return (ScratchFile)scratchFileOrLock;
        }
    }
    
    /**
     * @return The current size of the cache in MB.
     */
    public synchronized double getCurrentCacheSizeMB() {
        return this.currentCacheSizeMB;
    }
    
    /**
     * @return The number of pending announcements
     */
    public synchronized int getAnnouncementCount() {
        return this.announcedKeys.size();
    }

    /**
     * @return The number of entries currently stored in the cache. 
     */
    public synchronized int getEntryCount() {
        return this.entriesByKey.size();
    }

    /**
     * This function cleans up all cached files and should be called as
     * a blueprint <code>destroy-function</code>.
     */
    public synchronized void destroy() {
        
        log.info("Purging [{}] cache entries with a total size of [{}MB].",
                this.entriesByKey.size(),this.currentCacheSizeMB);
        
        for (Map.Entry<String,CacheEntry> e : this.entriesByKey.entrySet()) {
            
            e.getValue().getScratchFile().deleteAndClose();
        }
        
        this.entriesByKey.clear();
        this.currentCacheSizeMB = 0.0;
    }

    @Override
    public
    IScratchBucket getBucket(String key) throws InterruptedException {
        
        return new ScratchBucketImpl(key,this.getScratchFile(key));
    }
    
    /**
     * @return The maximal number of megabytes in the cache.
     */
    public double getCacheSizeMB() {
        return this.cacheSizeMB;
    }

    /**
     * @return The maximal number of millisecond to wait for the creation
     *         of a resource by another thread.
     */
    public long getMaxWaitMillis() {
        return this.maxWaitMillis;
    }

    /**
     * @return The configured file deleter.
     */
    public IFileDeleter getFileDeleter() {
        return this.fileDeleter;
    }

    /**
     * @param cacheSizeMB The maximal size of the cache in MegaBytes.
     */
    public void setCacheSizeMB(double cacheSizeMB) {
        this.cacheSizeMB = cacheSizeMB;
    }

    /**
     * @param maxWaitMillis The maximal time to wait for other threads
     *           to provide resources in milliseconds.
     */
    public void setMaxWaitMillis(long maxWaitMillis) {
        this.maxWaitMillis = maxWaitMillis;
    }

	/**
	 * @param fileDeleter The file deleter instance used to purge
	 *                    cache entries.
	 */
	public void setFileDeleter(IFileDeleter fileDeleter) {
		this.fileDeleter = fileDeleter;
	}
    
}
