/***********************************************************
 * $Id$
 *
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.10.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.security.auth.Destroyable;

import org.aopalliance.intercept.Joinpoint;
import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.sched.HasCallback;
import org.clazzes.util.sched.IJobStatus;
import org.clazzes.util.sched.IOneTimeScheduler;
import org.clazzes.util.sched.ITimedJob;
import org.clazzes.util.sched.JoinpointCallableAdapter;
import org.clazzes.util.sched.api.ILoggingCallback;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A bean, which implements {@link IOneTimeScheduler}.
 * In the context of blueprint, it can be used like this:
 *
 * <pre>
 *  &lt;bp:bean id="executorService" class="java.util.concurrent.Executors" factory-method="newFixedThreadPool"&gt;
 *     &lt;bp:argument value="3"/&gt;
 *   &lt;/bp:bean&gt;
 *
 *   &lt;bp:bean id="oneTimeScheduler" class="org.clazzes.util.sched.impl.OneTimeSchedulerImpl" init-method="start" destroy-method="shutdownNow"&gt;
 *      &lt;bp:property name="executorService" ref="executorService"/&gt;
 *      &lt;bp:property name="resultLifeTime" value="10000"/&gt;
 *      &lt;bp:property name="gcInterval" value="10000"/&gt;
 *   &lt;/bp:bean&gt;
 *  </pre>
 *
 *  Then inject the oneTimeScheduler as a bean into your service or servlet, write a Callable or Runnable, and call <code>scheduleJob</code> on it.
 *  Probably you also want to write a service for querying the state of your scheduled jobs, using <code>getJobState</code>.
 *  In that case, transmit the uuid returned by <code>scheduleJob</code> to the client, and use it as a parameter for the querying
 *  service method lateron.
 */
public class OneTimeSchedulerImpl implements IOneTimeScheduler {

    private static final Logger log = LoggerFactory.getLogger(OneTimeSchedulerImpl.class);

    private class WrappedCallable<V> implements Callable<V> {

        private final Callable<V> delegate;
        private final JobStatusImpl status;
        private final Map<String,Object> threadLocalValues;
        private final ITimedJob timedJob;

        public WrappedCallable(Callable<V> delegate, ITimedJob timedJob,
                JobStatusImpl status, Map<String,Object> threadLocalValues) {
            super();
            this.delegate = delegate;
            this.timedJob = timedJob;
            this.status = status;
            this.threadLocalValues = threadLocalValues;
        }

        @Override
        public V call() {

            if (this.threadLocalValues != null) {
                for (Map.Entry<String,Object> e : this.threadLocalValues.entrySet()) {

                    ThreadLocalManager.bindResource(e.getKey(),e.getValue());
                }
            }

            this.status.setRunning(true);

            V ret = null;
            Long nextExecutionDelay = null;

            try {
                if (log.isDebugEnabled()) {
                    log.debug("WrappedCallable.call called for uuid " + this.status.getUUID().toString());
                }
                ret = this.delegate.call();
                if (log.isDebugEnabled()) {
                    log.debug("WrappedCallable.call completed for uuid " + this.status.getUUID().toString());
                }

                // Decide about next execution of the job, if it is a ITimedJob
                if (this.timedJob != null) {
                    nextExecutionDelay = OneTimeSchedulerImpl.this.submitScheduledJobIfNecessary(this);
                }
                this.status.setResult(ret,nextExecutionDelay);
            }
            catch(Throwable e) {
                log.error("WrappedCallable caught exception for uuid " + this.status.getUUID().toString(), e);

                // Decide about next execution of the job, if it is a ITimedJob
                if (this.timedJob != null) {
                    nextExecutionDelay = OneTimeSchedulerImpl.this.submitScheduledJobIfNecessary(this);
                }
                this.status.setException(e,nextExecutionDelay);
            }
            finally {
                if (this.threadLocalValues != null) {
                    for (String k : this.threadLocalValues.keySet()) {
                        ThreadLocalManager.unbindResource(k);
                    }
                }
            }


            return ret;
        }

        private JobStatusImpl getStatus() {
            return this.status;
        }

        private ITimedJob getTimedJob() {
            return this.timedJob;
        }

    };

    // configurable bean properties
    private ExecutorService executorService;
    private long gcInterval;
    private long resultLifeTime;
    private Map<String,Object> threadLocalValues;

    // internal objects
    private Map<UUID,JobStatusImpl> jobs;
    private ScheduledExecutorService gcService;
    // Is the garbage collector service an internally instantiated one?
    private boolean ownGcService;
    private ScheduledFuture<?> gcFuture;

    public OneTimeSchedulerImpl() {

        this.gcInterval = 60000L;
        this.resultLifeTime = 60000L;
    }

    private int getNumberOfRunningJobs() {
        int number = 0;
        for (JobStatusImpl impl : this.jobs.values()) {
            if (!impl.isDone()) {
                number++;
            }
        }
        return number;
    }

    private int getNumberOfFinishedJobs() {
        int number = 0;
        for (JobStatusImpl impl : this.jobs.values()) {
            if (impl.isDone()) {
                number++;
            }
        }
        return number;
    }

    private Map<String,Object> propagateThreadLocals(Object delegate) {

        if (this.threadLocalValues == null) {
            return null;
        }

        Map<String,Object> ret = new HashMap<String, Object>(this.threadLocalValues.size());

        for (Map.Entry<String,Object> e : this.threadLocalValues.entrySet()) {

            Object v = e.getValue();

            if (v == null) {
                v = ThreadLocalManager.getBoundResource(e.getKey());

                if (v == null && ILoggingCallback.THREAD_LOCAL_KEY.equals(e.getKey())) {

                    v = CallbackHelper.getCallbackOfType(delegate,ILoggingCallback.class);
                }
            }

            if (v != null) {
                ret.put(e.getKey(),v);
            }
        }

        return ret;
    }

    /**
     * Close a job a per SCHEDUTIL-8
     *
     * @param jobStatus A job status just being garbage collected or
     *                  explicitly closed by the user.
     */
    private static void closeJob(UUID jobId, JobStatusImpl jobStatus) {

        Object res = jobStatus.getResult();

        if (res instanceof Closeable) {

            Closeable cr = (Closeable)res;

            try {

                if (log.isDebugEnabled()) {
                    log.debug("Closing closeable result [{}] of job [{}].",cr,jobId);
                }
                cr.close();
            }
            catch (IOException e) {
                log.warn("I/O error closing closeable result of job ["+jobId+"] upon garbage collection",e);
            }
        }

        jobStatus.destroyIfNotRunning();
    }

    private void gc() {

        long now = System.currentTimeMillis();

        synchronized (this) {

            if (log.isDebugEnabled()) {
                log.debug("Called gc() at " + now + ", with a total of " + this.jobs.size() + " existing ( "
                        + getNumberOfRunningJobs() + " running, " + getNumberOfFinishedJobs() + " finished)");
            }

            Iterator<Map.Entry<UUID,JobStatusImpl>> it = this.jobs.entrySet().iterator();

            while (it.hasNext()) {

                Entry<UUID,JobStatusImpl> entry = it.next();

                if (entry.getValue().isDone() && entry.getValue().getFinishedMillis() + this.resultLifeTime < now) {

                    if (log.isDebugEnabled()) {
                        log.debug("Garbage collecting job " + entry.getKey());
                    }
                    it.remove();
                    closeJob(entry.getKey(),entry.getValue());
                }
            }
        }
    }

    public void start() {

        if (log.isInfoEnabled()) {
            log.info("Called start()");
        }

        synchronized (this) {

            if (this.jobs != null) {
                throw new IllegalStateException("Try to start an already running one-time scheduler.");
            }
            this.jobs = new HashMap<UUID,JobStatusImpl>();

            if (this.gcService == null) {

                if (this.executorService instanceof ScheduledExecutorService) {
                    this.gcService = (ScheduledExecutorService)this.executorService;
                }
                else {
                    this.gcService = Executors.newSingleThreadScheduledExecutor();
                    this.ownGcService = true;
                }
            }


            this.gcFuture = this.gcService.scheduleWithFixedDelay(new Runnable() {
                @Override
                public void run() {
                    OneTimeSchedulerImpl.this.gc();
                }
            },this.gcInterval,this.gcInterval,TimeUnit.MILLISECONDS);
        }
    }

    private void cancelInternals() {
        this.jobs = null;

        if (this.ownGcService) {

            this.gcService.shutdownNow();
            this.gcService = null;
            this.gcFuture = null;
            this.ownGcService = false;
        }
        else {

            this.gcFuture.cancel(true);
            this.gcFuture = null;
        }
    }

    protected void shutdownGracefully(boolean hard) {

        Map<UUID, JobStatusImpl> jobsToCancel;

        synchronized (this) {

            if (this.jobs == null) {
                return;
            }

            if (log.isInfoEnabled()) {
                log.info("Called shutdownGracefully("+hard+"), with a total of " + this.jobs.size() + " existing ("
                        + getNumberOfRunningJobs() + " running, " + getNumberOfFinishedJobs() + " finished)");
            }

            jobsToCancel = this.jobs;

            this.cancelInternals();
        }

        for (Map.Entry<UUID,JobStatusImpl> je: jobsToCancel.entrySet()) {

            Future<?> future = je.getValue().getFuture();

            if (future != null && !future.isDone()) {
                try {
                    je.getValue().getFuture().cancel(hard);
                }
                catch (Throwable e) {
                    log.warn("Caught exception cancelling job ["+je.getKey()+"] upon shutdown of scheduler",e);
                }
            }

            closeJob(je.getKey(),je.getValue());
        }
    }

    /**
     * <p>Cancel all pending jobs scheduled by this instance.
     * Currently running jobs will not be interrupted.</p>
     *
     * <p>Warning: Unlike the semantics of <code>sched-util-1.2.0</code>
     * and earlier versions, this method will now shut down the underlying
     * executor instance.
     * </p>
     */
    public void shutdown() {

        this.shutdownGracefully(false);
    }

    public void shutdownNow() {

        this.shutdownGracefully(true);
    }

    @SuppressWarnings({ "rawtypes", "unchecked" })
    private static JobStatusImpl makeJobStatus(Object hint, UUID uuid) {

        JobStatusImpl ret;

        if (hint instanceof HasCallback<?>) {
            ret =new JobStatusWithCallbackImpl((HasCallback<?>)hint, uuid);
        }
        else {
            ret = new JobStatusImpl(uuid);
        }

        if (hint instanceof Destroyable) {
            ret.setDestroyable((Destroyable)hint);
        }

        return ret;
     }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#scheduleJob(java.lang.Runnable)
     */
    @Override
    public UUID scheduleJob(Runnable runnable) {

        return this.scheduleJobInner(runnable,Executors.callable(runnable));
    }



    @Override
    public UUID scheduleJob(Joinpoint joinpoint) {

        return this.scheduleJobInner(joinpoint,new JoinpointCallableAdapter(joinpoint));
    }

    /**
     * Submits a ITimedJob if its getNextExecutionDelay method indicates to do so.
     *
     * @param wrappedCallable Runnable implementing ITimedJob
     * @param status corresponding JobStatusImpl
     * @return The next execution delay or <code>null</code>, if the job has not
     *         actually been scheduled or the job does not implement {@link ITimedJob}.
     */
    private Long submitScheduledJobIfNecessary(WrappedCallable<?> wrappedCallable) {

        if (wrappedCallable.getTimedJob() == null) {

            wrappedCallable.getStatus().setFuture(this.executorService.submit(wrappedCallable));
            return null;
        }
        else {
            if (this.executorService instanceof ScheduledExecutorService) {

                ScheduledExecutorService executorService = (ScheduledExecutorService)this.executorService;

                Long nextExecutionDelay = null;

                try {

                    nextExecutionDelay = wrappedCallable.getTimedJob().getNextExecutionDelay();
                } catch (Throwable e) {
                    log.error("Caught exception calling ITimedJob.getNextExecutionDelay() on Job with uuid ["+
                            wrappedCallable.getStatus().getUUID()+
                            "], job will not be scheduled",e);
                }

                if (nextExecutionDelay != null) {
                    if (log.isDebugEnabled()) {
                        log.debug("Scheduling job with uuid [{}] with delay [{}s].",
                                wrappedCallable.getStatus().getUUID(),nextExecutionDelay.doubleValue() * 0.001);
                    }
                    wrappedCallable.getStatus().setFuture(executorService.schedule(wrappedCallable, nextExecutionDelay, TimeUnit.MILLISECONDS));
                } else {

                    if (log.isInfoEnabled()) {
                        log.info("Not scheduling job with uuid [{}] since it does not wish to do so.",
                                wrappedCallable.getStatus().getUUID());
                    }
                    // don't set a null future to the status in order to let the
                    // user query the canceled/done state from the last future.
                }

                return nextExecutionDelay;
            }
            else {

                String msg = "If a ITimedJob is passed to a OneTimeScheduler, a ScheduledExecutorService has to be used.";
                log.error(msg);
                throw new IllegalArgumentException(msg);
            }
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#scheduleJob(java.util.concurrent.Callable)
     */
    @Override
    public <V> UUID scheduleJob(Callable<V> callable) {

        return this.scheduleJobInner(callable,callable);
    }

    @Override
    public <V> void scheduleJob(UUID id, Callable<V> callable) {

        this.scheduleJobInner(id,callable,callable);
    }

    private <V> UUID scheduleJobInner(Object delegate, Callable<V> callable) {
        UUID id = UUID.randomUUID();
        return this.scheduleJobInner(id,delegate,callable);
    }

    private <V> UUID scheduleJobInner(UUID id, Object delegate, Callable<V> callable) {

        if (log.isDebugEnabled()) {
            log.debug("Scheduling callable job [" + id + "].");
        }

        JobStatusImpl status = makeJobStatus(delegate, id);

        synchronized (this) {

            if (this.jobs == null) {
                throw new IllegalStateException("Try to schedule a job on a stopped one-time scheduler.");
            }

            JobStatusImpl conflict = this.jobs.putIfAbsent(id,status);

            if (conflict != null) {
                throw new IllegalArgumentException("Try to schedule job with duplicate ID ["+id+"]");
            }
        }

        ITimedJob timedJob = null;

        if (delegate instanceof ITimedJob) {

            timedJob = (ITimedJob)delegate;
        }

        WrappedCallable<V> wrappedCallable = new WrappedCallable<V>(callable,timedJob,status,this.propagateThreadLocals(delegate));

        Long nextExecutionDelay = this.submitScheduledJobIfNecessary(wrappedCallable);

        // for timed jobs, set a timestamp and an execution delay.
        if (nextExecutionDelay != null) {
            status.setResult(null,nextExecutionDelay);
        }

        return id;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#getAllJobsIds()
     */
    @Override
    public List<UUID> getAllJobsIds() {

        if (log.isDebugEnabled()) {
            log.debug("Called getAllJobsIds");
        }

        synchronized (this) {

            if (this.jobs == null) {
                return null;
            }

            Set<UUID> keys = this.jobs.keySet();

            List<UUID> ret = new ArrayList<UUID>(keys.size());
            ret.addAll(keys);
            return ret;
         }
    }

    private JobStatusImpl getJobStatusImpl(UUID jobId) {

        synchronized (this) {
            if (this.jobs == null) {
                return null;
            }

            return this.jobs.get(jobId);
         }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#getJobStatus(java.util.UUID)
     */
    @Override
    public IJobStatus getJobStatus(UUID jobId) {

        synchronized (this) {

            if (this.jobs == null) {
                return null;
            }

            if (log.isDebugEnabled()) {
                log.debug("Called getJobStatus for job " + jobId + "(" + this.jobs.size() + " existing ("
                    + getNumberOfRunningJobs() + " running, " + getNumberOfFinishedJobs() + " finished)");
            }

            return this.jobs.get(jobId);
         }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#waitForFinish(java.util.UUID)
     */
    @Override
    public IJobStatus waitForFinish(UUID jobId) throws InterruptedException, ExecutionException {

        if (log.isDebugEnabled()) {
            log.debug("Called waitForFinish() for job " + jobId);
        }

        JobStatusImpl status = this.getJobStatusImpl(jobId);

        if (status != null) {
            Future<?> future = status.getFuture();
            future.get();
        }

        return status;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#waitForFinish(java.util.UUID, long)
     */
    @Override
    public IJobStatus waitForFinish(UUID jobId, long timeoutMillis) throws InterruptedException, ExecutionException, TimeoutException {
        if (log.isDebugEnabled()) {
            log.debug("Called waitForFinish() for job " + jobId);
        }

        JobStatusImpl status = this.getJobStatusImpl(jobId);

        if (status != null) {
            Future<?> future = status.getFuture();
            future.get(timeoutMillis,TimeUnit.MILLISECONDS);
        }

        return status;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#cancelJob(java.util.UUID, boolean)
     */
    @Override
    public IJobStatus cancelJob(UUID jobId, boolean mayInterrupt) {
        if (log.isDebugEnabled()) {
            log.debug("Called cancelJob for job " + jobId + ", mayInterrupt is "+ mayInterrupt);
        }

        JobStatusImpl status = this.getJobStatusImpl(jobId);

        if (status != null) {
            Future<?> future = status.getFuture();
            future.cancel(mayInterrupt);
        }

        return status;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#purgeResult(java.util.UUID)
     */
    @Override
    public IJobStatus purgeResult(UUID jobId) {

        if (log.isDebugEnabled()) {
            log.debug("Called purgeResult for job " + jobId);
        }

        synchronized (this) {
            JobStatusImpl status = this.jobs == null ? null : this.jobs.get(jobId);

            if (status != null) {

                if (status.isDone()) {

                    this.jobs.remove(jobId);
                    closeJob(jobId,status);

                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("==> Not purging job [{}] since it is not yet done.",jobId);
                    }
                }
            }

            return status;
         }
    }

    /**
     * @return the executorService
     */
    public ExecutorService getExecutorService() {
        return this.executorService;
    }

    /**
     * @param executorService the executorService to set
     */
    public void setExecutorService(ExecutorService executorService) {
        this.executorService = executorService;
    }



    public ScheduledExecutorService getGcService() {
        return this.gcService;
    }

    /**
     * Set a scheduled executor service for garbage collection.
     * If the garbage collector service has not been set, either
     * the executor service provided by {@link #setExecutorService(ExecutorService)}
     * is used, if this service implements {@link ScheduledExecutorService}
     * or an internal single-threaded executor service is internally used.
     *
     * @param gcService The new scheduled exectuor service used for the garbage collector.
     */
    public void setGcService(ScheduledExecutorService gcService) {

        if (this.ownGcService) {
            throw new IllegalStateException("Cannot override internal garbage collector service after start() has been called.");
        }

        this.gcService = gcService;
    }

    /**
     * @return the garbage collection interval in milliseconds.
     */
    public long getGcInterval() {
        return this.gcInterval;
    }

    /**
     * @param gcInterval the gcInterval to set
     */
    public void setGcInterval(long gcInterval) {
        this.gcInterval = gcInterval;
    }

    /**
     * @return the lifetime of results after a job has finished in milliseconds.
     */
    public long getResultLifeTime() {
        return this.resultLifeTime;
    }

    /**
     * @param resultLifeTime the lifetime of results after a job has finished in
     *                       milliseconds to set.
     */
    public void setResultLifeTime(long resultLifeTime) {
        this.resultLifeTime = resultLifeTime;
    }

    /**
     * @return A list of thread local value to propagate to the
     *         jobs.
     */
    public Map<String, Object> getThreadLocalValues() {
        return this.threadLocalValues;
    }

    /**
     * @param threadLocalValues A list of thread local value to propagate to the
     *         jobs to set. If the value for a given key is <code>null</code>,
     *         the thread local value for that key of the scheduling thread is
     *         propagated to the asynchronous job.
     */
    public void setThreadLocalValues(Map<String, Object> threadLocalValues) {
        this.threadLocalValues = threadLocalValues;
    }

}
