/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.11.2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.util.ArrayList;
import java.util.List;

import org.clazzes.util.sched.api.ILoggingCallback;
import org.clazzes.util.sched.api.LogMessage;
import org.clazzes.util.sched.api.LogPriority;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A callback for long-running jobs.
 */
public class LoggingCallbackImpl implements ILoggingCallback {

    private static final Logger log = LoggerFactory.getLogger(LoggingCallbackImpl.class); 
    private static final int DEFAULT_MAX_CACHED_MESSAGES = 1000;
    
    private final int maxCachedMessages;
    
    public LoggingCallbackImpl(int maxCachedMessages) {
        
        this.maxCachedMessages = maxCachedMessages;
    }
    
    public LoggingCallbackImpl() {
        
        this.maxCachedMessages = DEFAULT_MAX_CACHED_MESSAGES;
    }

    private List<LogMessage> logMessages;
	private boolean hasOverflown;
	
	protected synchronized void addMessage(LogPriority pri, String msg) {
		
	    if (this.hasOverflown) {
	        return;
	    }
	    
	    if (this.logMessages == null) {
	        this.logMessages = new ArrayList<LogMessage>(this.maxCachedMessages);
	    }
	    else {
	        
	        if (this.logMessages.size() >= this.maxCachedMessages) {
	            
	            log.warn("More than [{}] messages cached in memory, did your client starve?",this.maxCachedMessages);
	            this.hasOverflown = true;
	            return;
	        }
	    }
	    
	    LogMessage lm = new LogMessage();
	    lm.setTimestamp(System.currentTimeMillis());
	    lm.setPriority(pri);
	    lm.setMessage(msg);
	    
	    this.logMessages.add(lm);
	}

    @Override
    public void ok(String msg) {
        this.addMessage(LogPriority.OK,msg);
    }
	
	@Override
	public void debug(String msg) {
		
		this.addMessage(LogPriority.DEBUG,msg);
	}

	@Override
	public void info(String msg) {

		this.addMessage(LogPriority.INFO,msg);
	}

	@Override
	public void warning(String msg) {
		
		this.addMessage(LogPriority.WARN,msg);
	}

	@Override
	public void error(String msg) {
		
		this.addMessage(LogPriority.ERROR,msg);		
	}

    /**
     * @return The configured size of the message cache. 
     */
    public int getMaxCachedMessages() {
        return this.maxCachedMessages;
    }

    /**
	 * @return The log messages accumulated since the last call
	 *         to this function.
	 */
	@Override
    public synchronized List<LogMessage> getLastLogMessages() {
		
		try {
			return this.logMessages;
		}
		finally {
			this.logMessages = null;
			this.hasOverflown = false;
		}
	}
	
}
