/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.10.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.util.UUID;

import org.clazzes.util.aop.ThreadLocalManager;
import org.clazzes.util.aop.i18n.Messages;
import org.clazzes.util.sched.IJobStatus;
import org.clazzes.util.sched.IOneTimeScheduler;
import org.clazzes.util.sched.api.IJobStatusService;
import org.clazzes.util.sched.api.ILoggingCallback;
import org.clazzes.util.sched.api.JobStatusDTO;

/**
 * A service for delivering job-statii
 */
public class JobStatusServiceImpl implements IJobStatusService {

	private IOneTimeScheduler oneTimeScheduler;
	private boolean allowCancelTimedJob;
	
	private static final JobStatusDTO makeJobStatusDTO(IJobStatus status) {
		
		JobStatusDTO ret = null;
		
		if (status != null) {
			
			ret = new JobStatusDTO();
			ret.setDone(status.isDone());
			
			ret.setFinishedMillis(status.getFinishedMillis());
			ret.setNextExecutionDelay(status.getNextExecutionDelay());
			ret.setResult(status.getResult());
			
			if (status.getException() != null) {
				
				String msg = status.getException().toString();
				
				ret.setExceptionMessage(msg);
			}
			
			ILoggingCallback cb = CallbackHelper.getCallbackOfType(status,ILoggingCallback.class);
			
            if (cb != null) {
                
                ret.setLastLogMessages(cb.getLastLogMessages());
            }
            
		}
		
		return ret;
	}
	
	@Override
	public JobStatusDTO getJobStatus(UUID jobId) {
	
		IJobStatus status = this.oneTimeScheduler.getJobStatus(jobId);
		
		return makeJobStatusDTO(status);
	}

	@Override
    public JobStatusDTO waitForJob(UUID jobId, long timeout) {
	    
	    IJobStatus status;
	            
	    try {
            status = this.oneTimeScheduler.waitForFinish(jobId,timeout);
            
        } catch (Exception e) {
            
            status = this.oneTimeScheduler.getJobStatus(jobId);
        }
	    
        return makeJobStatusDTO(status);
    }

    @Override
	public JobStatusDTO cancelJob(UUID jobId) {
		
	    IJobStatus status;

	    if (this.allowCancelTimedJob) {
	        
	        status = this.oneTimeScheduler.cancelJob(jobId,true);
	        
            ILoggingCallback cb = CallbackHelper.getCallbackOfType(status,ILoggingCallback.class);

            if (cb != null) {
                
                Messages i18n = SchedUtilMessages.getMessages(ThreadLocalManager.getLoginLocale()); 
                cb.warning(i18n.formatString("jobInterruptedByUser",jobId));
            }
	    }
	    else {
	        
	        status = this.oneTimeScheduler.getJobStatus(jobId);
	        
	        if (status.getNextExecutionDelay() != null) {

	            ILoggingCallback cb = CallbackHelper.getCallbackOfType(status,ILoggingCallback.class);

	            if (cb != null) {
	                
	                Messages i18n = SchedUtilMessages.getMessages(ThreadLocalManager.getLoginLocale()); 
	                cb.warning(i18n.formatString("timedJobCancelDisabled",jobId));
	            }
	        }
	        else {
	            status = this.oneTimeScheduler.cancelJob(jobId,true);

	            ILoggingCallback cb = CallbackHelper.getCallbackOfType(status,ILoggingCallback.class);

	            if (cb != null) {
                    
                    Messages i18n = SchedUtilMessages.getMessages(ThreadLocalManager.getLoginLocale()); 
                    cb.warning(i18n.formatString("jobInterruptedByUser",jobId));
                }
	        }
	    }
    	
		return makeJobStatusDTO(status);
	}
	
	/**
	 * @return The configure one time scheduler
	 */
	public IOneTimeScheduler getOneTimeScheduler() {
        return this.oneTimeScheduler;
    }

    /**
     * @param oneTimeScheduler The one time scheduler to use.
     */
    public void setOneTimeScheduler(IOneTimeScheduler oneTimeScheduler) {
		this.oneTimeScheduler = oneTimeScheduler;
	}

    /**
     * @return Whether cancelling of timed jobs is allowed.
     */
    public boolean isAllowCancelTimedJob() {
        return this.allowCancelTimedJob;
    }

    /**
     * @param allowCancelTimedJob Set whether cancelling of timed jobs
     *          is allowed.
     */
    public void setAllowCancelTimedJob(boolean allowCancelTimedJob) {
        this.allowCancelTimedJob = allowCancelTimedJob;
    }

}
