/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.10.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.util.UUID;
import java.util.concurrent.Future;

import javax.security.auth.DestroyFailedException;
import javax.security.auth.Destroyable;

import org.clazzes.util.sched.IJobStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *  The actual implementation of a job status.
 */
 class JobStatusImpl implements IJobStatus {

    private static final Logger log = LoggerFactory.getLogger(JobStatusImpl.class);

    private Future<?> future = null;
    private Object result = null;
    private Throwable exception = null;
    private boolean running;
    private long finishedMillis = -1L;
    private Long nextExecutionDelay;
    private UUID uuid;
    private Destroyable destroyable;
    
    public JobStatusImpl(UUID uuid) {
        super();
        this.uuid = uuid;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IJobStatus#isCancelled()
     */
    @Override
    public boolean isCancelled() {
        
        if (this.future == null) {
            return false;
        }
        else {
            return this.future.isCancelled();
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IJobStatus#isDone()
     */
    @Override
    public synchronized boolean isDone() {
        
        if (this.running) {
            return false;
        }
        
        if (this.future == null) {
            return false;
        }
        else {
            return this.future.isDone();
        }
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IJobStatus#getResult()
     */
    @Override
    public synchronized Object getResult() {
        
        return this.result;
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IJobStatus#getException()
     */
    @Override
    public synchronized Throwable getException() {
        
        return this.exception;
    }
    
    public synchronized void setRunning(boolean running) {
        this.running = running;
    }

    /**
     * @param result the result of the job to set.
     */
    protected synchronized void setResult(Object result, Long nextExecutionDelay) {
        this.result = result;
        this.running = false;
        this.finishedMillis = System.currentTimeMillis();
        this.nextExecutionDelay = nextExecutionDelay;
        this.destroyInternal(false);
    }

    /**
     * @param exception the exception to set
     */
    protected synchronized void setException(Throwable exception, Long nextExecutionDelay) {
        this.exception = exception;
        this.running = false;
        this.finishedMillis = System.currentTimeMillis();
        this.nextExecutionDelay = nextExecutionDelay;
        this.destroyInternal(false);
    }

    /**
     * Destroy the internal destroyable, if this job is currently not running.
     * This is needed in order to avvoid races with setExcetion fdurinig garbage
     * collection of job results.
     */
    protected void destroyIfNotRunning() {
        if (!this.running) {
            this.destroyInternal(true);
        }
    }

    @Override
    public synchronized Long getNextExecutionDelay() {
        return this.nextExecutionDelay;
    }

    /**
     * @param future the future to set
     */
    public void setFuture(Future<?> future) {
        this.future = future;
    }

    @Override
    public long getFinishedMillis() {
        return this.finishedMillis;
    }

    /**
     * @return The underlying future.
     */
    public Future<?> getFuture() {
        return this.future;
    }
    
    public UUID getUUID() {
        return this.uuid;
    }

    /**
     * @return the destroyable used for cleanup of job grabage collection.
     */
    protected Destroyable getDestroyable() {
        return this.destroyable;
    }
    
    /**
     * Select the destroyable to be detroyed on job garbage collection.
     * @param destroyable The destroyable instance, which is the job itself, if it
     *              implements the destroyable interface.
     */
    protected void setDestroyable(Destroyable destroyable) {
        this.destroyable = destroyable;
    }

    /**
     * Destroy the internal destroyable without any mutex locks.
     *
     * @param leftOver If this is a left-over job for specific logging.
     */
    private void destroyInternal(boolean leftOver) {

        if (this.destroyable!=null && !this.destroyable.isDestroyed()) {
            log.info("Destroying job ["+this.uuid+"] " +
                     (leftOver ? "upon garbage collection." : "upon termination."));
            try {
                this.destroyable.destroy();
            } catch (DestroyFailedException e) {
                log.warn("Failure destroying job ["+this.uuid+"]",e);
            }
        }
    }
}
