/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 05.10.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.impl;

import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

import javax.security.auth.Destroyable;

import org.aopalliance.intercept.Joinpoint;
import org.clazzes.util.aop.ProxyFactory;
import org.clazzes.util.sched.HasCallback;
import org.clazzes.util.sched.IJobStatus;
import org.clazzes.util.sched.IOneTimeScheduler;
import org.clazzes.util.sched.ITimedJob;

/**
 * <p>A one-time scheduler, which delegates to another scheduler and schedules
 * {@link Runnable} and {@link Callable} instances intercepted by the provided
 * {@link ProxyFactory} instance.</p>
 * 
 * <p>This class takes care, that the {@link HasCallback} and {@link ITimedJob}
 * aspects of the passed runnables or callables are passed to the underlying
 * scheduler. Thereby, the {@link HasCallback} aspect is not intercepted whereas
 * the {@link ITimedJob} aspect is intercepted based on the setting of
 * {@link #setInterceptNextExecutionDelay(boolean)}.
 * </p>
 */
public class InterceptedOneTimeScheduler implements IOneTimeScheduler {

    private IOneTimeScheduler delegate;
    private ProxyFactory proxyFactory;
    private boolean interceptNextExecutionDelay;
    
    
    private Object interceptJob(Object job, Class<?> base) {
        
        // ultra-verbose version of assembling two arrays
        // of interfaces according to the aspects of the passed original...
        
        // count the culprits...
        int nifaces = 1;
        int nintercepted = 1;
        
        if (job instanceof HasCallback) {
            ++nifaces;
        }

        if (job instanceof Destroyable) {
            ++nifaces;
        }

        if (job instanceof ITimedJob) {
            ++nifaces;
            
            if (this.interceptNextExecutionDelay) {
                ++nintercepted;
            }
        }
        
        // ... allocate arrays ...
        Class<?>[] ifaces = new Class[nifaces];
        Class<?>[] intercepted = new Class[nintercepted];
        
        // ... fill in the culprits.
        nifaces = 0;
        nintercepted = 0;
        
        ifaces[nifaces++] = base;
        intercepted[nintercepted++] = base;
        
        if (job instanceof HasCallback) {
            ifaces[nifaces++] = HasCallback.class;
        }

        if (job instanceof Destroyable) {
            ifaces[nifaces++] = Destroyable.class;
        }

        if (job instanceof ITimedJob) {
            
            ifaces[nifaces++] = ITimedJob.class;
            
            if (this.interceptNextExecutionDelay) {
                intercepted[nintercepted++] = ITimedJob.class;
            }
        }
        
       return this.proxyFactory.getMultiTypedProxy(job,ifaces,intercepted);
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#scheduleJob(java.lang.Runnable)
     */
    @Override
    public UUID scheduleJob(Runnable runnable) {
        
    	Runnable proxy = (Runnable)this.interceptJob(runnable,Runnable.class);
         
        return this.delegate.scheduleJob(proxy);        
    }

    @Override
    public UUID scheduleJob(Joinpoint joinpoint) {

        Joinpoint proxy = (Joinpoint)this.interceptJob(joinpoint,Joinpoint.class);
        
        return this.delegate.scheduleJob(proxy);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#scheduleJob(java.util.concurrent.Callable)
     */
    @Override
    public <V> UUID scheduleJob(Callable<V> callable) {

        @SuppressWarnings("unchecked")
        Callable<V> proxy = (Callable<V>)this.interceptJob(callable,Callable.class);

        return this.delegate.scheduleJob(proxy);
    }

    @Override
    public <V> void scheduleJob(UUID id, Callable<V> callable) {

        @SuppressWarnings("unchecked")
        Callable<V> proxy = (Callable<V>)this.interceptJob(callable,Callable.class);

        this.delegate.scheduleJob(id,proxy);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#getAllJobsIds()
     */
    @Override
    public List<UUID> getAllJobsIds() {
        
        return this.delegate.getAllJobsIds();
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#getJobStatus(java.util.UUID)
     */
    @Override
    public IJobStatus getJobStatus(UUID jobId) {
        
        return this.delegate.getJobStatus(jobId);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#waitForFinish(java.util.UUID)
     */
    @Override
    public IJobStatus waitForFinish(UUID jobId) throws InterruptedException,
            ExecutionException {
        
        return this.delegate.waitForFinish(jobId);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#waitForFinish(java.util.UUID, long)
     */
    @Override
    public IJobStatus waitForFinish(UUID jobId, long timeoutMillis)
            throws InterruptedException, ExecutionException, TimeoutException {
        
        return this.delegate.waitForFinish(jobId,timeoutMillis);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#cancelJob(java.util.UUID, boolean)
     */
    @Override
    public IJobStatus cancelJob(UUID jobId, boolean mayInterrupt) {
        
        return this.delegate.cancelJob(jobId,mayInterrupt);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sched.IOneTimeScheduler#purgeResult(java.util.UUID)
     */
    @Override
    public IJobStatus purgeResult(UUID jobId) {
        
        return this.delegate.purgeResult(jobId);
    }

    /**
     * @return the delegate one-time scheduler.
     */
    public IOneTimeScheduler getDelegate() {
        return this.delegate;
    }

    /**
     * @param delegate the delegate one-time scheduler to set.
     */
    public void setDelegate(IOneTimeScheduler delegate) {
        this.delegate = delegate;
    }

    /**
     * @return the proxy factory used to intercept Runnables and Callables.
     */
    public ProxyFactory getProxyFactory() {
        return this.proxyFactory;
    }

    /**
     * @param proxyFactory the proxy factory used to intercept Runnables and
     *             Callables to set. It should have an already configured
     *             list of interceptors.
     */
    public void setProxyFactory(ProxyFactory proxyFactory) {
        this.proxyFactory = proxyFactory;
    }

    /**
     * @return Whether the method {@link ITimedJob#getNextExecutionDelay()} is intercepted or not.
     *         This may be set to <code>true</code>, if you want to schedule
     *         a timed job and the calculation of the next execution delay
     *         needs some resources like database connection or the like.
     */
    public boolean isInterceptNextExecutionDelay() {
        return this.interceptNextExecutionDelay;
    }

    /**
     * @param interceptNextExecutionDelay Set whether the method
     *         {@link ITimedJob#getNextExecutionDelay()} is intercepted or not.
     *         This may be set to <code>true</code>, if you want to schedule
     *         a timed job and the calculation of the next execution delay
     *         needs some resources like database connection or the like.
     */
    public void setInterceptNextExecutionDelay(boolean interceptNextExecutionDelay) {
        this.interceptNextExecutionDelay = interceptNextExecutionDelay;
    }

    
}
