package org.clazzes.util.sched.functional;

import org.clazzes.util.sched.ITimedJob;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.TimeZone;

/*
 * Calculates the timed job to be called in a day.
 * Important: this does not account for downtime at the scheduled time slot!
 *
 * Somewhat copied from CDES, heavily modified in MDA and stripped down to its core for DDS.
 */
public class DailyTimed implements ITimedJob {

    /**
     * Grace a second from the desired start time (today), to avoid confusion in case the job
     * is started a bit before the scheduled time (getNextExecutionDelay works in a relative manner,
     * I don't know how exact that is), and completes very fast. (old comment)
     */
    private static final long SCHEDULE_GRACE_MILLI = 1000L; // 1 second, so that a schedule may take a time
    private static final TimeZone DEFAULT_TIMEZONE = TimeZone.getTimeZone("Europe/Vienna");

    private final LocalTime timeToCall;

    private static final Logger log = LoggerFactory.getLogger(DailyTimed.class);
    private final TimeZone timeZone;

    public DailyTimed(LocalTime timeToCall, TimeZone timeZone) {
        assert timeToCall != null;
        this.timeZone = timeZone != null ? timeZone : DEFAULT_TIMEZONE;
        this.timeToCall = timeToCall;
    }

    public Long getNextExecutionDelay() {
        // Return value given in ms

        Instant now = Instant.now(); // do once, so that it is not killed at 00:00:00
        long nowMillis = now.toEpochMilli();

        //we probably do not need to convert into epochMillis, but could use compareTo instead
        //LocalDate currentDate = LocalDate.ofInstant(now, timeZone.toZoneId()); // JDK 9+
        LocalDate currentDate = now.atZone(timeZone.toZoneId()).toLocalDate();   // JDK 8
        long timeToCallTodayMillis = this.timeToCall
                .atDate(currentDate)
                .atZone(timeZone.toZoneId())
                .toInstant()
                .toEpochMilli();

        if (timeToCallTodayMillis > nowMillis + SCHEDULE_GRACE_MILLI) {
            // If the desired start time today is not yet reached, schedule the job to be started today

            long retValue = timeToCallTodayMillis - nowMillis;
            if (log.isDebugEnabled()) {
                String todayDt = this.timeToCall.atDate(currentDate).toString();
                log.debug(this + ": scheduling the job to be started today at '" + todayDt
                        + "', now is '" + now + "'; returning [" + retValue + "]");
            }
            return retValue;
        }

        // Else the job needs to wait until tomorrow.
        LocalDate tomorrowDate = currentDate.plusDays(1);
        long timeToCallTomorrowMillis = this.timeToCall
                .atDate(tomorrowDate)
                .atZone(timeZone.toZoneId())
                .toInstant()
                .toEpochMilli();

        if (timeToCallTomorrowMillis < nowMillis) {
            String tomorrowDt = this.timeToCall.atDate(tomorrowDate).toString();
            // If the job start tomorrow is in the past, something very weird has happened.
            throw new RuntimeException(this + ": startNextDay = '" + tomorrowDt
                    + "' is expected to be after '" + now + "' but was not.");
        }

        // Else just schedule the job to be started tomorrow at the desired time, as specified by startSeconds
        long retValue = timeToCallTomorrowMillis - nowMillis;
        if (log.isDebugEnabled()) {
            String tomorrowDt = this.timeToCall.atDate(tomorrowDate).toString();
            log.debug(this + ": scheduling the job to be started tomorrow, at '" + tomorrowDt
                    + "', now is '" + now + "'; returning [" + retValue + "]");
        }

        return retValue;
    }

    @Override
    public String toString() {
        return "Daily at " + timeToCall;
    }
}
