/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 13.05.2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.cache;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;

import org.clazzes.util.aop.IFileDeleter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A temporary file for download stored as result of background operations
 * like printing of ZIP file export.
 */
public class ScratchFile implements Closeable {

    private static final Logger log = LoggerFactory.getLogger(ScratchFile.class);
    
    private final File file;
    private final String mimeType;
    private final String disposition;
    private IFileDeleter fileDeleter;
    private boolean closed;
    
    /**
     * Create a new scratch file, which will be deleted by {@link File#delete()}.
     * If you want your file robustly deleted, use the constructor
     * {@link ScratchFile#ScratchFile(IFileDeleter, File, String, String)}.
     * 
     * @param file The file to be delivered
     * @param mimeType The MIME type of the content
     * @param disposition The content-disposition a known from the likewise HTTP header.
     */
    public ScratchFile(File file, String mimeType, String disposition) {
        this.file = file;
        this.mimeType = mimeType;
        this.disposition = disposition;
        this.closed = false;
    }

    /**
     * Create a new scratch file, which will be deleted by
     * {@link IFileDeleter#deleteFile(File)}.
     * 
     * @param fileDeleter A file deleter used to robustly delete the scratch file. 
     * @param file The file to be delivered
     * @param mimeType The MIME type of the content
     * @param disposition The content-disposition a known from the likewise HTTP header.
     */
    public ScratchFile(IFileDeleter fileDeleter, File file, String mimeType, String disposition) {
        this.fileDeleter = fileDeleter;
        this.file = file;
        this.mimeType = mimeType;
        this.disposition = disposition;
        this.closed = false;
    }

    /**
     * @return The path where the data is physically stored.
     */
    public File getFile() {
        return this.file;
    }

    /**
     * @return The MIME type of the stored data.
     */
    public String getMimeType() {
        return this.mimeType;
    }
    
    /**
     * @return The content-disposition, which is non-null for downloads.
     */
    public String getDisposition() {
        return this.disposition;
    }

    /**
     * @return Whether this scratch file is closed and not usable
     *         anymore.
     */
    public boolean isClosed() {
        return this.closed;
    }

    /**
     * No-throw variant for closing this object and deleting the underlying
     * file on the file system.
     */
    public void deleteAndClose() {
        
        if (!this.closed) {
            
            if (this.fileDeleter == null) {
            
                if (!this.file.delete()) {
                    log.warn("Cannot delete scratch file [{}].",this.file);
                }
            }
            else {
                this.fileDeleter.deleteFile(this.file);
                // the file deleter singleton will not be used any more,
                // give the garbage collector a chance.
                this.fileDeleter = null;
            }
            
            this.closed = true;
        }
    }
    
    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException {
        this.deleteAndClose();
    }

    public void finalize() throws Throwable {
        this.deleteAndClose();
    }

    @Override
    public String toString() {
        return "ScratchFile [file=" + this.file + ", mimeType=" + this.mimeType
                + ", disposition=" + this.disposition + "]";
    }
}
