/***********************************************************
 * $Id$
 *
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 16.02.2016
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sched.cache;

/**
 * <p>The interface implemented by a scratch file cache.
 * A cache should be provided by this interface to other
 * service components.</p>
 *
 * <p>The preferred usage pattern of a cache is by the java7
 * try-with-resource pattern like this:</p>
 * <pre>
 *     IScratchFileCache scratchFileCache;
 *     String key = "myapp-fancy-jpg-84120461";
 *
 *     try (IScratchBucket bucket = scratchFileCache.getBucket(key)) {
 *
 *         if (bucket.getScratchFile() == null) {
 *
 *             File tmp = File.createTempFile("myapp-tmp-",".jpg"));
 *
 *             // fill tmp with contents...
 *
 *             bucket.provideScratchFile(tmp,"image/jpeg","inline");
 *         }
 *
 *         // deliver contents of bucket.getScratchFile()
 *     }
 * </pre>
 */
public interface IScratchFileCache {

    /**
     * Return a scratch bucket for the given key.
     * If the bucket has no data, you may add data by
     * {@link IScratchBucket#provideScratchFile(java.io.File, String, String)}.
     *
     * In any way, the returned bucket has to be closed by the thread calling
     * this method.
     *
     * @param key The application-specific cache key.
     * @return The bucket belonging to the given key. The calling thread is obliged
     *         to call {@link IScratchBucket#close()} on the returned bucket.
     * @throws InterruptedException If the thread has been interrupted while
     *             waiting for another thread to provide data for the given key.
     */
    IScratchBucket getBucket(String key) throws InterruptedException;
}
