/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.11.2013
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched.api;

import java.io.Serializable;
import java.util.List;

/**
 * A status of a long-running job to be transferred to a client
 * over a remote API.
 */
public class JobStatusDTO implements Serializable {

	private static final long serialVersionUID = -2925125458278787418L;

	private boolean done;
	private long finishedMillis;
	private Long nextExecutionDelay;
	private String exceptionMessage;
	private Object result;
	private List<LogMessage> lastLogMessages;

	/**
	 * @return Whether the job has been finished. If an exception occurred,
	 *         {@link #getExceptionMessage()} return a non-<code>null</code>
	 *         message.
	 */
	public boolean isDone() {
		return this.done;
	}
	/**
	 * @param done The finished flag to set.
	 */
	public void setDone(boolean done) {
		this.done = done;
	}
	/**
	 * @return A message describing the exception, which stopped the job.
	 *         If <code>null</code>, the job finished normally
	 */
	public String getExceptionMessage() {
		return this.exceptionMessage;
	}
	/**
	 * @param exceptionMessage The message describing the exception,
	 *           which stopped the job.
	 * 
	 */
	public void setExceptionMessage(String exceptionMessage) {
		this.exceptionMessage = exceptionMessage;
	}
	/**
	 * @return The log messages accumulated since the last status query
	 *         or <code>null</code>, if no messages have been logged in
	 *         the meanwhile.
	 */
	public List<LogMessage> getLastLogMessages() {
		return this.lastLogMessages;
	}
	/**
	 * @param lastLogMessages The log messages accumulated since the last
	 *         status query to set.
	 */
	public void setLastLogMessages(List<LogMessage> lastLogMessages) {
		this.lastLogMessages = lastLogMessages;
	}
	/**
	 * @return The termination timestamp of the job, if {@link #isDone()}
	 *         returns <code>true</code>.
	 */
	public long getFinishedMillis() {
		return this.finishedMillis;
	}
	/**
	 * @param finishedMillis The termination timestamp of the job to set.
	 */
	public void setFinishedMillis(long finishedMillis) {
		this.finishedMillis = finishedMillis;
	}
	
    /**
     * @return The next intended execution since {@link #getFinishedMillis()}
     *         in milliseconds.
     *         If <code>null</code>, this is a one-time job or a timed job,
     *         which is finished and will not be executed again.
     *         If {@link #isDone()} returns <code>false</code>, this was the
     *         intended next executio since the last execution.
     */
    public Long getNextExecutionDelay() {
        return this.nextExecutionDelay;
    }
    /**
     * @param nextExceutionDelay The next execution delay for a timed job to set.
     */
    public void setNextExecutionDelay(Long nextExceutionDelay) {
        this.nextExecutionDelay = nextExceutionDelay;
    }
    /**
     * @return The result of the job, if {@link #isDone()} returns
     *         <code>true</code>.
     */
    public Object getResult() {
        return this.result;
    }
    /**
     * @param result The result of the job to set. Please assure,
     *          that the result is a serializable by your RPC framework.
     */
    public void setResult(Object result) {
        this.result = result;
    }
	
}
