/***********************************************************
 * $Id$
 * 
 * scheduler utilities of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 04.10.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sched;

import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeoutException;

import org.aopalliance.intercept.Joinpoint;
import org.clazzes.util.aop.ReflectiveMethodInvocation;

/**
 * An scheduler, which is based on an {@link Executor} and executes
 * one-time tasks.
 */
public interface IOneTimeScheduler {

    /**
     * <p>Schedule the start of a job that returns no result.</p>
     * 
     * <p>The passed instance may implement {@link ITimedJob} in
     * which case, the job is called repeatedly according to the
     * results of {@link ITimedJob#getNextExecutionDelay()}.</p>
     * 
     * <p>If <code>runnable</code> moreover implements {@link HasCallback},
     * the job status will be {@link IJobStatusWithCallback} instance,
     * which bears additional application-specific information.</p>
     * 
     * @param runnable The runnable to schedule.
     * @return the uuid of the new job
     */
    public UUID scheduleJob(Runnable runnable);
    
    /**
     * <p>Schedule the start of a job that returns a result.</p>
     * 
     * <p>The passed instance may implement {@link ITimedJob} in
     * which case, the job is called repeatedly according to the
     * results of {@link ITimedJob#getNextExecutionDelay()}.</p>
     * 
     * <p>If <code>callable</code> moreover implements {@link HasCallback},
     * the job status will be {@link IJobStatusWithCallback} instance,
     * which bears additional application-specific information.</p>
     * 
     * @param callable The callable to be scheduled.
     * @return the uuid of the new job
     */
    public <V> UUID scheduleJob(Callable<V> callable);
    
    /**
     * <p>Schedule the start of a job that returns a result with a user-supplied
     * unique ID.</p>
     *
     * <p>The passed instance may implement {@link ITimedJob} in
     * which case, the job is called repeatedly according to the
     * results of {@link ITimedJob#getNextExecutionDelay()}.</p>
     *
     * <p>If <code>callable</code> moreover implements {@link HasCallback},
     * the job status will be {@link IJobStatusWithCallback} instance,
     * which bears additional application-specific information.</p>
     *
     * @param id the user-supplied job ID, which should be unique
     * @param callable The callable to be scheduled.
     * @throws IllegalArgumentException if the given ID has already been used by an active job.
     * @see UUID#randomUUID()
     */
    public <V> void scheduleJob(UUID id, Callable<V> callable);

    /**
     * <p>Schedule the start of a job that returns a result.</p>
     * 
     * <p>This method exists in order to schedule AOP instances like
     * {@link ReflectiveMethodInvocation}. If you have the choice,
     * please implement {@link Callable} in favor of {@link Joinpoint},
     * because this avoids an internal indirection layer.</p>
     * 
     * <p>The passed instance may implement {@link ITimedJob} in
     * which case, the job is called repeatedly according to the
     * results of {@link ITimedJob#getNextExecutionDelay()}.</p>
     * 
     * <p>If <code>joinpoint</code> moreover implements {@link HasCallback},
     * the job status will be {@link IJobStatusWithCallback} instance,
     * which bears additional application-specific information.</p>
     * 
     * @param joinpoint The joinpoint to schedule.
     * @return the uuid of the new job
     */
    public UUID scheduleJob(Joinpoint joinpoint);

    /** 
     * @return a list of all currently existing job ids.  This may include jobs that are
     *  already completed.
     */
    public List<UUID> getAllJobsIds();
    
    /**
     * Query the job status, if the scheduled job implemented {@link HasCallback},
     * the returned status will be an instance of {@link IJobStatusWithCallback}.
     * 
     * @param jobId The ID of the job as returned by {@link #scheduleJob(Callable)}
     *              or {@link #scheduleJob(Runnable)}.
     * @return a job status or <code>null</code> if no such job status exists (e.g.
     *         because the job has completed and was already garbage collected, or
     *         because the scheduler was shut down)
     */
    public IJobStatus getJobStatus(UUID jobId);
    
    /** Waits until the given job has finished.  Returns the status of the job, or
     *  <code>null</code> if no such job exists, e.g. because the scheduler does no
     *  longer run.
     * 
     * @param jobId job id
     * @return job status as described, <code>null</code> if no such job status exists
     * @throws InterruptedException
     * @throws ExecutionException
     */
    public IJobStatus waitForFinish(UUID jobId) throws InterruptedException, ExecutionException;
    
    /** Waits until the given job has finished, but no longer than the given timeout.
     *  Returns the status of the job, or <code>null</code> if no such job exists, 
     *  e.g. because the scheduler does no longer run.  
     * 
     * @param jobId job id
     * @param timeoutMillis timeout
     * @return job status as described, <code>null</code> if no such job status exists
     * @throws InterruptedException
     * @throws ExecutionException
     * @throws TimeoutException
     */
    public IJobStatus waitForFinish(UUID jobId, long timeoutMillis) throws InterruptedException, ExecutionException, TimeoutException;
    
    /** Cancels the given job.  Returns the status of the job, or <code>null</code> if no such job exists, 
     *  e.g. because the scheduler does no longer run. 
     * 
     * @param jobId job id
     * @param mayInterrupt <code>true</code> if and only if the job is allowed to be interrupted
     * @return job status as described, <code>null</code> if no such job status exists
     */
    public IJobStatus cancelJob(UUID jobId, boolean mayInterrupt);
    
    /** Purges the results of the given job.
     * 
     * @param jobId job id
     * @return job status, <code>null</code> if no such job status exists
     */
    public IJobStatus purgeResult(UUID jobId);
}
