/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.xml;

import java.util.Stack;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

/**
 * This class implements a SAX content handler, which holds a stack
 * of TagHandlers and thus eases the writing of SAX parsers.
 * 
 * @author wglas
 */
public abstract class SimpleContentHandler implements ContentHandler {

    private Stack<SAXTagHandler> handlers;
    
    /**
     * This function is called, when the root element is opened
     * through {@link ContentHandler#startElement(String, String, String, Attributes)}.
     * 
     * @param uri The namespace URI of the received tag. 
     * @param localName The name of the tag without any namespace prefix.
     * @param name The fully-qualified name of the tag. This is the tag name,
     *             if the namespace URI is null. 
     * @param atts Any attributes attached to the child 
     * @return The handler, which processes the child. A null return value
     *         tells the SAX parser to ignore the content of the child.
     * @throws SAXException Upon errors processing the data.
     */
    abstract protected SAXTagHandler getRootHandler(String uri, String localName, String name,
                                                    Attributes atts) throws SAXException;
    
    /**
     * Construct a SimpleContentHandler.   
     */
    public SimpleContentHandler()
    {
        this.handlers = new Stack<SAXTagHandler>();
    }
    
    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] ch, int start, int length)
            throws SAXException {
        
        SAXTagHandler handler = this.handlers.peek();
        
        if (handler != null)
            handler.processCharacters(ch,start,length);
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#endDocument()
     */
    public void endDocument() throws SAXException {
       
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String name)
            throws SAXException {
        
        SAXTagHandler handler = this.handlers.pop();
        
        if (handler != null)
            handler.closeTag(uri,localName,name);
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#endPrefixMapping(java.lang.String)
     */
    public void endPrefixMapping(String prefix) throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#ignorableWhitespace(char[], int, int)
     */
    public void ignorableWhitespace(char[] ch, int start, int length)
            throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#processingInstruction(java.lang.String, java.lang.String)
     */
    public void processingInstruction(String target, String data)
            throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator locator) {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#skippedEntity(java.lang.String)
     */
    public void skippedEntity(String name) throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#startDocument()
     */
    public void startDocument() throws SAXException {
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String name,
            Attributes atts) throws SAXException {
        
        SAXTagHandler childHandler;
        
        if (this.handlers.size() <= 0)
        {
            childHandler = this.getRootHandler(uri,localName,name,atts);
        }
        else
        {
            SAXTagHandler handler = this.handlers.peek();
        
            if (handler != null)
                childHandler = handler.startChildElement(uri,localName,name,atts);
            else
                childHandler = null;
        } 
            
        this.handlers.push(childHandler);
    }

    /* (non-Javadoc)
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String, java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri)
            throws SAXException {
    }

}
