/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 08.12.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.xml;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

/**
 * An interface, which allows to implement SAX parsers as a stack of
 * tag handlers.
 * 
 * @author wglas
 */
public interface SAXTagHandler {
    
    /**
     * This function is called, when character data inside the processed
     * tag is received through {@link ContentHandler#characters(char[], int, int)}.
     * 
     * @param ch The character array.
     * @param start The start index inside the character array of the received data.
     * @param length The length of the received data.
     * @throws SAXException Upon errors processing the data.
     */
    public void processCharacters(char[] ch, int start, int length)
                                  throws SAXException;
    
    /**
     * This function is called, when a child element is opened inside the processed
     * tag through {@link ContentHandler#startElement(String, String, String, Attributes)}.
     * 
     * @param uri The namespace URI of the received tag. 
     * @param localName The name of the tag without any namespace prefix.
     * @param name The fully-qualified name of the tag. This is the tag name,
     *             if the namespace URI is null. 
     * @param atts Any attributes attached to the child 
     * @return The handler, which processes the child. A null return value
     *         tells the SAX parser to ignore the content of the child.
     * @throws SAXException Upon errors processing the data.
     */
    public SAXTagHandler startChildElement(String uri, String localName, String name,
                                           Attributes atts) throws SAXException;
    
    /**
     * This function is called, when the tag is closed.
     * 
     * @param uri The namespace URI of the tag. 
     * @param localName The name of the tag without any namespace prefix.
     * @param name The fully-qualified name of the tag. This is the tag name,
     *             if the namespace URI is null. 
     * @throws SAXException Upon errors processing the data.
     */
    public void closeTag(String uri, String localName, String name) throws SAXException; 
}
