/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2007-12-07
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.xml;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.UncheckedIOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

/**
 * This implementation of LSInput represents an XML input source
 * for module-resident schema or DTD descriptions, which are emitted
 * by a handcrafted instance of {@link LSResourceResolver}.
 */
public class ModuleLSInput implements LSInput {

    private static final Logger log = LoggerFactory.getLogger(ModuleLSInput.class);

    private final String resourcePath;
    private final Module module;
    private String baseURI;
    private String publicId;
    private String systemId;
    private boolean certifiedText;

    /**
     * Construct a module-resident LSInput instance.
     *
     * @param module The module from which to retrieve the resource stream.
     * @param resourcePath The path, where the resource is located. Will be passed
     *                     to {@link Module#getResourceAsStream(String)}.
     * @param publicId The XML public ID of the resource.
     * @param systemId The XML system ID of the resource.
     * @param baseURI The XML base URI of the resource.
     * @param certifiedText The XML-1.1 flag, which marks the resource as certified.
     */
    public ModuleLSInput(Module module, String resourcePath,
            String publicId,
            String systemId,
            String baseURI,
            boolean certifiedText) {

        this.resourcePath = resourcePath;
        this.publicId = publicId;
        this.systemId = systemId;
        this.baseURI = baseURI;
        this.certifiedText = certifiedText;
        this.module = module;
    }

    /**
     * Construct a module-resident LSInput instance.
     *
     * @param module The module from which to retrieve the resource stream.
     * @param resourcePath The path, where the resource is located. Will be passed
     *                     to {@link Module#getResourceAsStream(String)}.
     * @param publicId The XML public ID of the resource.
     * @param systemId The XML system ID of the resource.
     * @param baseURI The XML base URI of the resource.
     */
    public ModuleLSInput(Module module, String resourcePath,
            String publicId,
            String systemId,
            String baseURI) {

        this.resourcePath = resourcePath;
        this.publicId = publicId;
        this.systemId = systemId;
        this.baseURI = baseURI;
        this.certifiedText = true;
        this.module = module;
    }

    /**
     * Construct a module-resident LSInput instance. This constructor loads from
     * the module of the {@link ModuleLSInput} class itself.
     *
     * @param resourcePath The path, where the resource is located. Will be passed
     *                     to {@link Module#getResourceAsStream(String)}.
     * @param publicId The XML public ID of the resource.
     * @param systemId The XML system ID of the resource.
     * @param baseURI The XML base URI of the resource.
     * @param certifiedText The XML-1.1 flag, which marks the resource as certified.
     */
    public ModuleLSInput(String resourcePath,
            String publicId,
            String systemId,
            String baseURI,
            boolean certifiedText) {

        this.resourcePath = resourcePath;
        this.publicId = publicId;
        this.systemId = systemId;
        this.baseURI = baseURI;
        this.certifiedText = certifiedText;
        this.module = ModuleLSInput.class.getModule();
    }

    /**
     * Construct a module-resident LSInput instance. This constructor loads from
     * the module of the {@link ModuleLSInput} class itself.
     *
     * @param resourcePath The path, where the resource is located. Will be passed
     *                     to {@link Module#getResourceAsStream(String)}.
     * @param publicId The XML public ID of the resource.
     * @param systemId The XML system ID of the resource.
     * @param baseURI The XML base URI of the resource.
     */
    public ModuleLSInput(String resourcePath,
            String publicId,
            String systemId,
            String baseURI) {

        this.resourcePath = resourcePath;
        this.publicId = publicId;
        this.systemId = systemId;
        this.baseURI = baseURI;
        this.certifiedText = true;
        this.module = ModuleLSInput.class.getModule();
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getBaseURI()
     */
    public String getBaseURI() {
        return this.baseURI;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getByteStream()
     */
    public InputStream getByteStream() {

        InputStream is;

        try {
            if (log.isDebugEnabled()) {
                log.debug("Loading resource [{}] from module [{}].",
                        this.resourcePath,this.module);
            }

            is = this.module.getResourceAsStream(this.resourcePath);
        } catch (IOException e) {
            throw new UncheckedIOException("Unable to load resource ["+
                        this.resourcePath+"] from module ["+
                        this.module+"]",e);
        }

        if (is == null) {
            log.warn("Cannot find resource [{}] in module [{}].",
                        this.resourcePath,this.module);
        }

        return is;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getCertifiedText()
     */
    public boolean getCertifiedText() {
        return this.certifiedText;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getCharacterStream()
     */
    public Reader getCharacterStream() {
        return null;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getEncoding()
     */
    public String getEncoding() {
        return null;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getPublicId()
     */
    public String getPublicId() {
        return this.publicId;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getStringData()
     */
    public String getStringData() {
        return null;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#getSystemId()
     */
    public String getSystemId() {
        return this.systemId;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setBaseURI(java.lang.String)
     */
    public void setBaseURI(String baseURI) {
        this.baseURI = baseURI;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setByteStream(java.io.InputStream)
     */
    public void setByteStream(InputStream byteStream) {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setCertifiedText(boolean)
     */
    public void setCertifiedText(boolean certifiedText) {
        this.certifiedText = certifiedText;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setCharacterStream(java.io.Reader)
     */
    public void setCharacterStream(Reader characterStream) {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setEncoding(java.lang.String)
     */
    public void setEncoding(String encoding) {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setPublicId(java.lang.String)
     */
    public void setPublicId(String publicId) {
        this.publicId = publicId;
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setStringData(java.lang.String)
     */
    public void setStringData(String stringData) {
        throw new UnsupportedOperationException();
    }

    /* (non-Javadoc)
     * @see org.w3c.dom.ls.LSInput#setSystemId(java.lang.String)
     */
    public void setSystemId(String systemId) {
        this.systemId = systemId;
    }


}
