/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sec;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.apache.commons.codec.binary.Base64;

/**
 * The SSHA1 hasher
 */
public class SSHA1PasswordHasher extends PlainPasswordHasher
{
    // apache commons-codec Base64 is a Base64-encoder/decoder which is widely available,
    // and thus preferred to sun.misc.Base64Encoder
    private Base64 base64encoder;

	public SSHA1PasswordHasher()
	{
		super();
		this.base64encoder = new Base64();
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#hashPassword(java.lang.String, java.lang.String)
	 */
	@Override
    public String hashPassword(String cleartext, String salt)
	{
		return this.algo_prefix + sha1(cleartext, salt);
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#checkPassword(java.lang.String, java.lang.String)
	 */
	@Override
    public boolean checkPassword(String cleartext, String hashed)
	{
		String hash_salt = stripAlgorithmKey(hashed);
		int l = hash_salt.length(); // the naked hash is 20 bytes long, but
									// base64 makes it longer!
		String salt = hash_salt.substring(l - 4);
		String rehashed = hashPassword(cleartext, salt);
		if (rehashed.equals(hashed)) return true;

		// check for pre 0.9.2 bug (double hash in hashed)
		if (hashed.length() == rehashed.length() + 4 &&
		        hash_salt.charAt(l-8) == salt.charAt(0) &&
		        hash_salt.charAt(l-7) == salt.charAt(1) &&
		        hash_salt.charAt(l-6) == salt.charAt(2) &&
		        hash_salt.charAt(l-5) == salt.charAt(3)   ) {

		    hashed = hashed.substring(0,hashed.length()-4);
		    return rehashed.equals(hashed);
		}

        // check for pre 0.9.2 bug (double hash in hashed preceded by '\r\n')
        if (hashed.length() == rehashed.length() + 6  &&
                hash_salt.charAt(l-10) == '\r' &
                hash_salt.charAt(l-9)  == '\n' &&
                hash_salt.charAt(l-8)  == salt.charAt(0) &&
                hash_salt.charAt(l-7)  == salt.charAt(1) &&
                hash_salt.charAt(l-6)  == salt.charAt(2) &&
                hash_salt.charAt(l-5)  == salt.charAt(3)) {

            hashed = hashed.substring(0,hashed.length()-10) + salt;
            return rehashed.equals(hashed);
        }


		return false;
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#getAlgorithmName()
	 */
	@Override
    public String getAlgorithmName()
	{
		return "SSHA1";
	}

	private String sha1(String cleartext, String salt)
	{
		try
		{
			// convert input to utf-8 based byte[]
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			OutputStreamWriter osw = new OutputStreamWriter(baos, "UTF-8");
			osw.write(cleartext);
			osw.write(salt);
			osw.close();
			baos.close();

			// make sha1 stuff
			MessageDigest sha1digest = MessageDigest.getInstance("SHA-1");
			sha1digest.update(baos.toByteArray());
			byte[] sha1ba = sha1digest.digest();
			String s = this.base64encoder.encodeToString(sha1ba);

			return s.trim() + salt;
		} catch (NoSuchAlgorithmException e)
		{
			throw new RuntimeException(e);
		} catch (UnsupportedEncodingException e)
		{
			throw new RuntimeException(e);
		} catch (IOException e)
		{
			throw new RuntimeException(e);
		}
	}


    /* (non-Javadoc)
     * @see org.clazzes.util.sec.PasswordHasher#getDefaultSaltLength()
     */
    public int getSaltLength() {
        return 4;
    }


}
