/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 28.03.2007
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sec;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

/**
 * @author lech
 *
 * This factory creates password hashers.
 */
public class HasherFactory implements PasswordHasherFactory {

    private final Map<String,PasswordHasher> hashers_byname;

    /**
     * Create an new instance knowing all clazzes-util hash implementations.
     */
    public HasherFactory() {

        this.hashers_byname = new HashMap<String, PasswordHasher>();
        PasswordHasher plain_hasher = new PlainPasswordHasher();
        this.hashers_byname.put(plain_hasher.getAlgorithmName().toUpperCase(Locale.ENGLISH), plain_hasher);
        SSHA1PasswordHasher sha1_hasher = new SSHA1PasswordHasher();
        this.hashers_byname.put(sha1_hasher.getAlgorithmName().toUpperCase(Locale.ENGLISH), sha1_hasher);
        CryptPasswordHasher crypt_hasher = new CryptPasswordHasher();
        this.hashers_byname.put(crypt_hasher.getAlgorithmName().toUpperCase(Locale.ENGLISH), crypt_hasher);
        BCryptPasswordHasher bcrypt_hasher = new BCryptPasswordHasher();
        this.hashers_byname.put(bcrypt_hasher.getAlgorithmName().toUpperCase(Locale.ENGLISH), bcrypt_hasher);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.PasswordHasherFactory#getPasswordHasher(java.lang.String)
     */
    public PasswordHasher getPasswordHasher(String algo_name)
    {
        //checkHashers();
        if (algo_name==null)
          return null;
        PasswordHasher hasher = this.hashers_byname.get(algo_name.toUpperCase(Locale.ENGLISH));
        return hasher;
    }
    
    /* (non-Javadoc)
     * @see org.clazzes.util.sec.PasswordHasherFactory#getKnownPasswordHashers()
     */
    public Collection<PasswordHasher> getKnownPasswordHashers() {
        // checkHashers();
        return Collections.unmodifiableCollection(this.hashers_byname.values());
    }

}
