/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.sec;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A Tool to hash passwords and test the PasswordHasher classes
 */
public class Hasher {

    private static final Logger log = LoggerFactory.getLogger(Hasher.class);

    protected static PasswordHasherFactory passwordHasherFactory;

    public static void main(String[] args) throws java.lang.Exception {
        // if called as "program", we can perhaps do different things
        // make a simple stdout log
        /*
         * log = Logger.getLogger("org.clazzes.util.sec.Hasher");
         * BasicConfigurator.configure(); log.addAppender(new
         * ConsoleAppender()); log.setLevel(Level.DEBUG);
         */
        // parse eventual params
        // what to do
        passwordHasherFactory = new HasherFactory();
        String command = System.getenv("CMD");
        if ((command == null) || (command.length() == 0)) {
            command = "hash";
            log.info("Using default CMD hash");
        } else {
            log.info("Using given CMD " + command);
        }
        // password
        String pwcleartext = System.getenv("PW");
        if ((pwcleartext == null) || (pwcleartext.length() == 0)) {
            log.error("No PW given.");
            System.exit(2);
        } else {
            log.info("Using given PW " + pwcleartext);
        }
        // work ...
        if (command.equals("hash")) {
            // salt?
            String salt = System.getenv("SALT");
            if ((salt == null) || (salt.length() == 0)) {
                salt = HashTools.randomSalt();
                log.info("No SALT given, using random salt " + salt);
            } else {
                log.info("Using given SALT " + salt);
            }
            // alg
            String algo_name = System.getenv("ALG");
            if ((algo_name == null) || (algo_name.length() == 0)) {
                algo_name = "SSHA1"; // SSHA1
                log.info("No ALG given, using default algorithm " + algo_name);
            } else {
                log.info("Using given ALG " + algo_name);
            }
            // hasher
            PasswordHasher pwhasher = passwordHasherFactory
                    .getPasswordHasher(algo_name);
            if (pwhasher == null) {
                log.error("Could not find hasher for algorithm " + algo_name);
                System.exit(3);
            } else {
                log.info("Got PasswordHasher of class "
                        + pwhasher.getClass().getName());
                String hashed = pwhasher.hashPassword(pwcleartext, salt);
                log.info("Computed password to " + hashed);
                System.exit(0);
            }
        }
        if (command.equals("check")) {
            // hashed?
            String hashed = System.getenv("HASHED");
            if ((hashed == null) || (hashed.length() == 0)) {
                log.error("No HASHED pw given to check");
                System.exit(4);
            } else {
                log.info("Using HASHED " + hashed);
            }
            String algo_name = HashTools.parseAlorithmName(hashed);
            log.info("Parsed algo_name to " + algo_name);
            PasswordHasher pwhasher = passwordHasherFactory
                    .getPasswordHasher(algo_name);
            if (pwhasher == null) {
                log.error("Could not find hasher for algorithm " + algo_name);
                System.exit(5);
            } else {
                log.info("Got PasswordHasher of class "
                        + pwhasher.getClass().getName());
                boolean pwok = pwhasher.checkPassword(pwcleartext, hashed);
                log.info("The PW seems to be " + (pwok ? "OK" : "WRONG"));
                System.exit(0);
            }
        }
        log.error("What did you want? " + command + "?");
        System.exit(1);
    }

}
