/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/


package org.clazzes.util.sec;

import java.security.SecureRandom;

/**
 * A view [Password]Hasher helpers center
 *
 * For usage, see classes Hasher and PersonOperations
 */
public class HashTools
{

	/**
	 * The recommended default salt length.
     * But be warned, the best known and widely used crypt algorithm uses a 2 char salt only.
	 */
	public static final byte DEFAULT_SALTLENGTH = 4;
	public static final String CRYPT_SALT_CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ./";
	public static final String B64_SALT_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
	public static final String SMS_SALT_CHARS ="abcdefghkmnpqrstuvwxyz0123456789";

    private static final SecureRandom secureRandom = new SecureRandom();

    /*
     * public static final Random randomizer = secureRandom;
     *
     * This field was included for binary backward compatibility with
     * clazzes-util-1.5 and has been dropped in clazzes-util-2.x
     * releases.
     *
     * If you need a static secure random generator use
     * {@link #getSecureRandom()} instead.
     */

    /**
     * @return A static secure random instance.
     */
    public static final SecureRandom getSecureRandom() {
        return secureRandom;
    }

    /**
	 * @return a random salt with the default salt length.
	 */
	public static String randomSalt()
	{
		return randomSalt(DEFAULT_SALTLENGTH);
	}

    /**
     * Creates a random salt by drawing characters from the given string.
     *
     * @param length The length of the salt to generate.
     * @param saltChars The character set to draw from.
     * @return A string of the given length filled with random characters
     *         out of the given set.
     */
    public static String randomSalt(int length, String saltChars)
    {
        if (length <= 0 || length > 256)
            throw new RuntimeException("HashTools.randomSalt("+length+"): Illegal length "+length);

        char[] salt = new char[length];

        for (int i=0; i < length; ++i) {
            salt[i] = saltChars.charAt(secureRandom.nextInt(saltChars.length()));
        }
        return new String(salt);
    }

    /**
     * Creates a random salt with the given length with the character set
     * {@value #CRYPT_SALT_CHARS} specified by <code>crypt(3)</code>.
     *
	 * @param length the wanted length of the salt. must be &gt; 0 and &lt;= 256.
	 * @return the random salt
	 */
	public static String randomSalt(int length)
	{
        return randomSalt(length,CRYPT_SALT_CHARS);
	}

    /**
     * Creates a random salt with the given length with the character set
     * {@value #B64_SALT_CHARS} specified as the URL-safe base64 character
     * set in
     * <a href="https://tools.ietf.org/html/rfc7515#appendix-C">RFC 7515</a>.
     *
     * @param length the wanted length of the salt. must be &gt; 0 and &lt;= 256.
     * @return the URL-safe base64 random salt
     */
    public static String randomSaltBase64(int length)
    {
        return randomSalt(length,B64_SALT_CHARS);
    }

    /**
     * Creates a random salt with the given length with the character set
     * {@value #SMS_SALT_CHARS} containing 32 characters, which may be easily
     * distinguished from each other on a smart phone display.
     *
     * @param length the wanted length of the salt. must be &gt; 0 and &lt;= 256.
     * @return the URL-safe base64 random salt
     */
    public static String randomSaltSms(int length)
    {
        return randomSalt(length,SMS_SALT_CHARS);
    }

    /**
     * Parses the algorithm name from the given hashed password's algorithm prefix.
	 * @param from_hashed_password a hashed password including the algorithm prefix.
	 * @return the algorithm specified in the given hashed password's algorithm prefix. null in case of any parsing error.
	 */
	public static String parseAlorithmName(String from_hashed_password)
	{
		if (from_hashed_password == null)
			return null;
		int left_idx = from_hashed_password.indexOf('{');
		if (left_idx != 0)
			return null;
		int right_idx = from_hashed_password.lastIndexOf('}');
		if (right_idx < 0)
			return null;
        if (right_idx - left_idx < 2)
            return null;
		String algo_name = from_hashed_password.substring(left_idx + 1,
				right_idx);
		return algo_name;
	}

	/**
	 * Check the given password against the hashed password using one of the
	 * hashers registered in the given hasher factory.
	 *
	 * @param passwordHasherFactory The password hasher factory.
	 * @param givenPassword The plain text password to check.
	 * @param hashedPassword The hashed password with a hash type prefix.
	 * @return Whether the given password matches the hashed password.
	 */
	public static boolean checkPassword(PasswordHasherFactory passwordHasherFactory, String givenPassword, String hashedPassword) {
	    if (hashedPassword == null || givenPassword == null) {
	        return false;
	    }
	    final String algoName = HashTools.parseAlorithmName(hashedPassword);
	    if (algoName == null || algoName.length() == 0) {
	        return false;
	    }
	    final PasswordHasher hasher = passwordHasherFactory.getPasswordHasher(algoName);
	    if (hasher == null) {
	        return false;
	    }
	    if (hasher.checkPassword(givenPassword, hashedPassword)) {
	        return true;
	    }
	    return false;
	}
}
