/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 29.08.2012
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sec;

import java.util.List;

/**
 * A service interface for providing login facilities based on
 * domain/user/password information.
 */
public interface DomainPasswordLoginService {

    /**
     * The name of the property, which defines the login method.
     */
    public static final String LOGIN_MECHANISM_KEY = "login.mechanism";

    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #changePassword(String, String, String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_CHANGE_PASSWORD = 0x0001;
    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #sendPassword(String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_SEND_PASSWORD   = 0x0002;
    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #searchUser(String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_SEARCH_USER     = 0x0004;
    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #deactivateUser(String, String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_DEACTIVATE_USER = 0x0008;
    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #getGroups(String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_GET_GROUPS      = 0x0010;
    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #getGroupMembers(String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_GET_GROUPMEMBERS = 0x0020;

    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #getU2FCredentialIds(DomainPrincipal)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_GET_U2F_CREDENTIALS_IDS = 0x0040;

    /**
     * If this bit is set in the return value of {@link #getSupportedFeatures(String)},
     * the function {@link #verifyU2FCredential(DomainPrincipal, String, String)}
     * is supported by the current implementation.
     */
    public static final int FEATURE_VERIFY_U2F_CREDENTIAL = 0x0080;

    /**
     * @return An integer representing a bit mask of supported features.
     *
     * @param domain The domain for which to tell about the supported features. May be null. May get ignored if the domain is irrelevant for the feature list.
     * @see #FEATURE_CHANGE_PASSWORD
     * @see #FEATURE_SEND_PASSWORD
     * @see #FEATURE_SEARCH_USER
     * @see #FEATURE_DEACTIVATE_USER
     * @see #FEATURE_GET_GROUPS
     * @see #FEATURE_GET_GROUPMEMBERS
     * @see #FEATURE_GET_U2F_CREDENTIALS_IDS
     * @see #FEATURE_VERIFY_U2F_CREDENTIAL
     */
    public int getSupportedFeatures(String domain);

    /**
     * @return The default domain, which is used, if the user did not enter a domain name.
     */
    String getDefaultDomain();

    /**
     * @return The list of domains for which the service can perform login services.
     */
    public List<String> getDomains();

    /**
     * Try to log on to the a login backend.
     *
     * @param domain The domain of the user to log on.
     * @param username The username.
     * @param password The password.
     * @return A principal for the logged on user or <code>null</code>, if the user has not been
     *         successfully authenticated.
     */
    public DomainPrincipal tryLogin(String domain, String username, String password);

    /**
     * Change the password of a user.
     *
     * @param domain The domain of the user to change the password for.
     * @param username The username.
     * @param oldPassword The old password of the user.
     * @param newPassword The new password to set.
     * @throws SecurityException When the old password is wrong.
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public void changePassword(String domain, String username, String oldPassword, String newPassword);

    /**
     * This method generates a new password or send a "new password" link to the user.
     * Doing so is only supported by systems, which generally know the e-mail address of
     * a given user.
     * @param domain The domain of the user to log on.
     * @param username The username.
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public void sendPassword(String domain, String username);

    /**
     * Search a user in the user database.
     *
     * @param domain The domain to search.
     * @param username The exact username to match.
     * @return A principal for the user or <code>null</code>, if the user could
     *         not be found in the user database.
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public DomainPrincipal searchUser(String domain, String username);

    /**
     * Deactivate a user.
     *
     * @param domain The domain of the user to deactivate.
     * @param username The user name.
     * @param reason A reason like "Too many failed logon attempts" or "dictionary attack".
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public void deactivateUser(String domain, String username, String reason);

    /**
     * Get the groups of a user.
     *
     * @param domain The domain of the user to search the groups for.
     * @param userName The user to search the groups for.
     * @return A list of groups of which the user is a member.
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public List<DomainGroup> getGroups(String domain, String userName);

    /**
     * Get members of a group.
     *
     * @param domain The domain of the group to search for members.
     * @param groupName The group to get the members from.
     * @return A list of users which are members of the group.
     * @throws UnsupportedOperationException if this operation is not supported by the target system.
     */
    public List<DomainPrincipal> getGroupMembers(String domain, String groupName);

    /**
     * Return a list of U2F second-factor credentials IDs.
     *
     * @param principal A pre-authenticated principal to fetch the known U2F credential IDs for.
     * @return A list of credentials IDs used for the authenticate workflow.
     */
    public default List<String> getU2FCredentialIds(DomainPrincipal principal) {
        throw new UnsupportedOperationException();
    }

    /**
     * Verify an U2F assertion credential.
     *
     * @param principal A pre-authenticated principal to fetch the known U2F credential IDs for.
     * @param challenge The random challenge used for the authenticate U2F workflow.
     * @param credential the U2F assertion credential to be verified against the known attestation credentials
     *              for the given principal.
     * @return The provided principal with possible additional information on the authenticated token or <code>null</code>,
     *           if the verification failed.
     */
    public default DomainPrincipal verifyU2FCredential(DomainPrincipal principal, String challenge, String credential) {
        throw new UnsupportedOperationException();
    }
}
