/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.sec;

import org.apache.commons.codec.digest.Crypt;

/**
 * The Crypt hasher
 */
public class CryptPasswordHasher extends PlainPasswordHasher
{
    public enum Variant {
        CRYPT,
        SHA256,
        SHA512
    }

    private final Variant variant;

	public CryptPasswordHasher(Variant variant)
	{
		super();
		this.variant = variant;
	}

    public CryptPasswordHasher()
    {
       this(Variant.SHA512);
    }

    /* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#hashPassword(java.lang.String, java.lang.String)
	 */
	@Override
    public String hashPassword(String cleartext, String salt)
	{
		return this.algo_prefix + Crypt.crypt(cleartext,salt);
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#checkPassword(java.lang.String, java.lang.String)
	 */
	@Override
    public boolean checkPassword(String cleartext, String hashed)
	{
		String hash_salt = stripAlgorithmKey(hashed);

		String salt;

		if (hash_salt.startsWith("$5$") || hash_salt.startsWith("$6$")) {

		    int endSalt = hash_salt.indexOf('$',3);

		    if (endSalt < 0) {
		        throw new IllegalArgumentException("Crypt hash with SHA256 or SHA512 scheme does not contain a third dollar sign.");
		    }

		    salt = hash_salt.substring(0,endSalt);
		}
		else {
		    salt = hash_salt.substring(0,2);
		}

		String rehashed = this.hashPassword(cleartext, salt);
		return rehashed.equals(hashed);
	}

	/* (non-Javadoc)
	 * @see org.clazzes.util.sec.PlainPasswordHasher#getAlgorithmName()
	 */
	@Override
    public String getAlgorithmName()
	{
		return "CRYPT";
	}

    @Override
    public String hashPassword(String cleartext) {

        String salt;

        if (this.variant == Variant.CRYPT) {
            salt = HashTools.randomSalt(2);
        }
        else {

            String pfx = this.variant == Variant.SHA256 ? "$5$" : "$6$";

            salt = pfx + HashTools.randomSalt(19);
        }

        return this.hashPassword(cleartext,salt);
    }

    /* (non-Javadoc)
     * @see org.clazzes.util.sec.PasswordHasher#getDefaultSaltLength()
     */
    public int getSaltLength() {
        return  this.variant == Variant.CRYPT ? 2 : 19;
    }


}
