/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 01.11.2020
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.ryu;

import java.text.DecimalFormatSymbols;
import java.text.FieldPosition;
import java.text.Format;
import java.text.ParsePosition;
import java.util.Locale;

/**
 * A localized number format using the RYU algorithm for a specified locale.
 */
public class RyuNumberFormat extends Format {

    private static final long serialVersionUID = -1854037583417190194L;

    private final DecimalFormatSymbols symbols;
    private final RoundingMode roundingMode;
    private final int options;
    
    /**
     * @param symbols Custom decimal symbols used for formatting.
     * @param roundingMode The rounding mode to apply.
     * @param options Options a bit mask composed of options in {@link RyuOptions}.
     * @see RyuOptions#SHOW_DOT_ZERO
     * @see RyuOptions#SCIENTIFIC
     */
    public RyuNumberFormat(DecimalFormatSymbols symbols, RoundingMode roundingMode, int options) {
        this.symbols = symbols;
        this.roundingMode = roundingMode;
        this.options = options;
    }

    /**
     * Construct a number format with {@link RoundingMode#ROUND_EVEN}, custom
     * decimal symbols and custom options.
     * 
     * @param symbols Custom decimal symbols used for formatting.
     * @param options Options a bit mask composed of options in {@link RyuOptions}.
     * @see RyuOptions#SHOW_DOT_ZERO
     * @see RyuOptions#SCIENTIFIC
     */
    public RyuNumberFormat(DecimalFormatSymbols symbols,int options) {
        this(symbols,RoundingMode.ROUND_EVEN,options);
    }

    /**
     * Construct a number format with {@link RoundingMode#ROUND_EVEN} and custom
     * decimal symbols.
     * 
     * @param symbols Custom decimal symbols used for formatting.
     */
    public RyuNumberFormat(DecimalFormatSymbols symbols) {
        this(symbols,RoundingMode.ROUND_EVEN,RyuOptions.DEFAULT);
    }

    /**
     * Construct a number format with {@link RoundingMode#ROUND_EVEN} and standard
     * decimal symbols for the given locale.
     *
     * @param locale The local used for formatting numbers.
     */
    public RyuNumberFormat(Locale locale) {
        
        this(DecimalFormatSymbols.getInstance(locale));
    }

    /**
     * Construct a format with the standard decimal symbols for the given locale,
     * {@link RoundingMode#ROUND_EVEN} and custom options.
     * @param locale The local used for formatting numbers.
     * @param options Options a bit mask composed of options in {@link RyuOptions}.
     * @see RyuOptions#SHOW_DOT_ZERO
     * @see RyuOptions#SCIENTIFIC
     */
    public RyuNumberFormat(Locale locale, int options) {
        
        this(DecimalFormatSymbols.getInstance(locale),RoundingMode.ROUND_EVEN,options);
    }

    /**
     * Construct a format with the standard decimal symbols for the given locale.
     * @param locale The local used for formatting numbers.
     * @param roundingMode The rounding mode to apply.
     * @param options Options a bit mask composed of options in {@link RyuOptions}.
     * @see RyuOptions#SHOW_DOT_ZERO
     * @see RyuOptions#SCIENTIFIC
     */
    public RyuNumberFormat(Locale locale, RoundingMode roundingMode, int options) {
        
        this(DecimalFormatSymbols.getInstance(locale),roundingMode,options);
    }

    /* (non-Javadoc)
     * @see java.text.Format#format(java.lang.Object, java.lang.StringBuffer, java.text.FieldPosition)
     */
    @Override
    public StringBuffer format(Object obj, StringBuffer toAppendTo,
            FieldPosition pos) {
        
        Number n = (Number) obj;
        
        if (n instanceof Float) {
        
            RyuFloat.floatToString(toAppendTo,n.floatValue(),this.symbols,this.roundingMode,this.options);
        }
        else {
            RyuDouble.doubleToString(toAppendTo,n.doubleValue(),this.symbols,this.roundingMode,this.options);
        }
        
        return toAppendTo;
    }

    /* (non-Javadoc)
     * @see java.text.Format#parseObject(java.lang.String, java.text.ParsePosition)
     */
    @Override
    public Object parseObject(String source, ParsePosition pos) {
        
        throw new UnsupportedOperationException();
    }

}
