/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.reflect;


/**
 * This class provides some static function for transformting
 * property names to getter/setter names.
 * 
 * @author wglas
 */
public class PropertyHelper
{
    /**
     * Set to private, because we only provide for static functions.
     */
    private PropertyHelper()
    {}
    
    /**
     * @param property The property for which to retrieve the getter name.
     * @return The name of the getter method for the given property.
     * 
     * @see PropertyHelper#getBooleanGetterName(String)
     */
    public static String getGetterName(String property)
    {
        return "get" + Character.toUpperCase(property.charAt(0)) + property.substring(1);
    }
    
    /**
     * @param property The property for which to retrieve the setter name.
     * @return The name of the setter method for the given property.
     */
    public static String getSetterName(String property)
    {
        return "set" + Character.toUpperCase(property.charAt(0)) + property.substring(1);
    }
    
    /**
     * This method complements the functionality of {@link PropertyHelper#getGetterName(String)}
     * for boolen properties, where the getter method start with the string "is" instead of "get".
     * 
     * @param property The boolean property for which to retrieve the setter name.
     * @return The name of the setter method for the given property.
     */
    public static String getBooleanGetterName(String property)
    {
        return "is" + Character.toUpperCase(property.charAt(0)) + property.substring(1);
    }
    
    private static String extractProperty(String pfx, String methodName)
    {
        if (methodName == null ||
                methodName.length() < pfx.length()+1 ||
                !methodName.startsWith(pfx) ||
                !Character.isUpperCase(methodName.charAt(pfx.length())))
            return null;

        if (methodName.length() < pfx.length()+2 ||
                Character.isLowerCase(methodName.charAt(pfx.length()+1)))
        {
            // getXy -> property is xy
            return
            Character.toLowerCase(methodName.charAt(pfx.length())) +
            methodName.substring(pfx.length()+1);
        }
        else
        {
            // getXY -> property is XY.
            return methodName.substring(pfx.length());
        }
        
    }
    
    /**
     * Returns the property name of a given getter method name.
     * 
     * @param getterName The name of the getter function.
     * @return The associated property name or null, if the function name is not
     *         a valid getter name.
     *         
     * @see PropertyHelper#getBooleanPropertyFromGetterName(String)
     */
    public static String getPropertyFromGetterName(String getterName)
    {
       return extractProperty("get",getterName);
    }
    
    /**
     * Returns the property name of a given getter method name for boolean fields.
     * 
     * This method complements the functionality of {@link PropertyHelper#getGetterName(String)}
     * for boolen properties, where the getter method start with the string "is" instead of "get".
     *
     * @param getterName The name of the getter function.
     * @return The associated property name or null, if the function name is not
     *         a valid getter name.
     */
    public static String getBooleanPropertyFromGetterName(String getterName)
    {
       return extractProperty("is",getterName);
    }
    
    /**
     * Returns the property name of a given setter method name.
     * 
     * @param setterName The name of the setter function.
     * @return The associated property name or null, if the function name is not
     *         a valid setter name.
     */
    public static String getPropertyFromSetterName(String setterName)
    {
       return extractProperty("set",setterName);
    }
}
