/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 17.04.2009
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.reflect;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarInputStream;

/**
 * This class provides static functionality for handling java packages.
 * 
 * @author wglas
 */
public abstract class PackageHelper {

    /**
     * Return a list of classes, which reside in the same package an on the same
     * classpath entry than the given class.
     * 
     * Please note, that due to the nature of the java class loading mechanism it is
     * impossible to find all classes, which belong to a package, that is spread over several
     * jar files or classpath directories.
     * 
     * Furthermore it is impossible to implement this method without a given concrete class,
     * because some jar files do not contain a zip entry representing the package directory
     * itself. (Most notably, <code>rt.jar</code> from the JRE itself)
     * 
     * The returned list does not include inner classes, because these may be retrieved through
     * {@link Class#getClasses()} anyways.
     * 
     * @param clazz The class for which to retrieve all siblings.
     * @return A list of classes, which are on the same package than the given class.
     * @throws ClassNotFoundException
     */
    @SuppressWarnings("unchecked")
    public static List<Class> getPackageClasses(Class clazz)
            throws ClassNotFoundException {
        
        List<Class> classes = new ArrayList<Class>();
        
        // extract package name plus trailing dot.
        int ii = clazz.getName().lastIndexOf('.');       
        String pckgname = clazz.getName().substring(0,ii+1);
       
        try {
            
            // we consider only the original classloader of the given class.
            // Note: cld may be null, if we are faced with a class from
            //       JRE itself.
            ClassLoader cld = clazz.getClassLoader();
            
            // path of the resource of the class file.
            String clsPath = clazz.getName().replace('.', '/') + ".class";
            
            // fetch the URL
            URL resource =
                cld != null ?
                cld.getResource(clsPath) :
                ClassLoader.getSystemResource(clsPath);
            
            // The class exist, but could not be found as a .class resource...
            if (resource == null) {
                throw new ClassNotFoundException("No resource for [" + clsPath + "].");
            }
            
            // This class comes from a class path directory.
            if ("file".equals(resource.getProtocol())) {
            
                // the package directory is the parent of the class file.
                File clsFile = new File(resource.getFile());
                File directory = clsFile.getParentFile();
            
                if (directory != null && directory.exists()) {
                    // Get the list of the files contained in the package
                    String[] files = directory.list();
                    for (int i = 0; i < files.length; i++) {
                        // we are only interested in .class files
                        if (files[i].endsWith(".class") &&
                                // filter out internal classes.
                                files[i].indexOf('$') < 0 &&
                                // filter out package-info.class
                                files[i].indexOf('-') < 0) {
                            // removes the .class extension
                        classes.add(Class.forName(pckgname +
                                files[i].substring(0, files[i].length() - 6),
                                true,cld));
                        }
                    }
                } else {
                    throw new ClassNotFoundException("[" + pckgname
                            + "] does not appear to be a valid package");
                }
            // This class comes from is a jar file on the classpath
            }
            else if ("jar".equals(resource.getProtocol())) {
                
                int idx = resource.getPath().indexOf('!');
                
                String jarURL = idx < 0 ? resource.getPath() : resource.getPath().substring(0,idx);
                // directory prefix of zip entries for the questionable package.
                String jPath = pckgname.replace('.', '/');
                
                try {
                    URL jarResource =
                        new URL(jarURL);
                    
                    File jarFile = new File(jarResource.getFile());
                    
                    // try to use JarFile if possible, because it is more performant
                    // than JarInputStream.
                    if (jarFile.exists()) {
                        
                        JarFile jf = new JarFile(jarFile);
                        
                        try {
                            Enumeration<JarEntry> entries = jf.entries();
                            
                            while (entries.hasMoreElements()) {
                                
                                JarEntry e = entries.nextElement();
                                
                                if (e.getName().startsWith(jPath) &&
                                        // exclude class from subpackages.
                                        e.getName().indexOf('/',jPath.length()) < 0 &&
                                        // filter out .class resources
                                        e.getName().endsWith(".class") &&
                                        // filter out internal classes.
                                        e.getName().indexOf('$',jPath.length()) < 0 &&
                                        // filter out package-info.class
                                        e.getName().indexOf('-',jPath.length()) < 0) {
                                    
                                    classes.add(Class.forName(pckgname +
                                            e.getName().substring(jPath.length(),e.getName().length() - 6),
                                            true,cld));
                                }
                            }
                        }
                        finally {
                            jf.close();
                        }
                        
                    } else {
                        
                        // fallback method using JarInputStream.
                        // This might be needed, if the jar URL points to a http resource
                        // or the like.
                        JarInputStream jis = new JarInputStream(jarResource.openStream());
                        
                        try {
                            JarEntry e;
                            
                            while ((e = jis.getNextJarEntry()) != null) {
                                
                                if (e.getName().startsWith(jPath) &&
                                        e.getName().indexOf('/',jPath.length()) < 0 &&
                                        e.getName().endsWith(".class") &&
                                        e.getName().indexOf('$',jPath.length()) < 0 &&
                                        // filter out package-info.class
                                        e.getName().indexOf('-',jPath.length()) < 0) {
                                    
                                    classes.add(Class.forName(pckgname +
                                            e.getName().substring(jPath.length(),e.getName().length() - 6),
                                            true,cld));
                                }
                            }
                        }
                        finally {
                            jis.close();
                        }
                    }
                    
                } catch (MalformedURLException e) {
                    
                    throw new ClassNotFoundException("URL [" + resource +"] for package ["+ pckgname
                            + "] points to an invalid jar-file.");

                } catch (IOException e) {
                    
                    throw new ClassNotFoundException("Error reading jar file [" + resource +"] for package ["+ pckgname+"].",e);
                }
                
                
            }
            else if ("jrt".equals(resource.getProtocol())) {
             
                int idx = resource.getPath().lastIndexOf('/');
                
                FileSystem fs = FileSystems.newFileSystem(URI.create("jrt:/"),Collections.EMPTY_MAP);
                
                Path pkgPath = fs.getPath("modules",resource.getPath().substring(1,idx));

                try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(pkgPath)) {
                    for (Path path : directoryStream) {
                        
                        String fn = path.getFileName().toString();
                       
                        if (fn.endsWith(".class") &&
                                // filter out internal classes.
                                fn.indexOf('$') < 0 &&
                                // filter out package-info.class
                                fn.indexOf('-') < 0) {
                            // removes the .class extension
                        classes.add(Class.forName(pckgname +
                                fn.substring(0, fn.length() - 6),
                                true,cld));
                        }
                    }
                }
            }
            else {
                throw new ClassNotFoundException("URL [" + resource +"] for package ["+ pckgname
                        + "] is neither a directory nor a jar-file resource.");
            }
            
        } catch (NullPointerException x) {
            throw new ClassNotFoundException("["+pckgname+"] does not appear to be a valid package.",x);
        } catch (MalformedURLException e) {
            throw new ClassNotFoundException("["+pckgname+"] could not be reduced to a correct URL.",e);
        } catch (IOException e) {
            throw new ClassNotFoundException("Content of package ["+pckgname+"] could not be listed.",e);
        }
   
        return classes;
    }

}
