/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.reflect;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import javax.persistence.AttributeOverride;
import javax.persistence.AttributeOverrides;
import javax.persistence.Column;

/**
 * An implementation of {@link FieldAccessor} for plain old
 * java objects (POJOs).
 */
public class POJOFieldAccessor implements FieldAccessor {
    
    private Field field;
    private Method fieldGetter;
    private Method fieldSetter;
    private String fieldName;
    private FieldAccessor idAccesor;

    /**
     * Instantiates a field accessor for a field of a given POJO.
     */
    public POJOFieldAccessor(Field field) {
        super();
        this.field = field;

        String getterName;
        if (field.getType().equals(Boolean.TYPE)
                || field.getType().equals(Boolean.class))
            getterName = PropertyHelper.getBooleanGetterName(field.getName());
        else
            getterName = PropertyHelper.getGetterName(field.getName());

        this.fieldGetter = getFieldGetter(getterName);

        String setterName = PropertyHelper.getSetterName(field.getName());

        try {
            this.fieldSetter = field.getDeclaringClass().getMethod(setterName,
                    field.getType());

            if (Modifier.isStatic(this.fieldSetter.getModifiers()))
                this.fieldSetter = null;

        } catch (SecurityException e) {
            this.fieldSetter = null;
        } catch (NoSuchMethodException e) {
            this.fieldSetter = null;
        }

        this.fieldName = this.field.getName();
    }

    private Method getFieldGetter(String getterName) {
        Method getter = null;
        try {
            getter = this.field.getDeclaringClass().getMethod(getterName);

            if (Modifier.isStatic(getter.getModifiers()))
                getter = null;

        } catch (SecurityException e) {
            getter = null;
        } catch (NoSuchMethodException e) {
            if ((this.field.getType().equals(Boolean.TYPE) || this.field
                    .getType().equals(Boolean.class))
                    && getterName.startsWith("is"))
                return getFieldGetter(PropertyHelper.getGetterName(this.field
                        .getName()));
            else
                getter = null;
        }
        return getter;
    }

    /**
     * Instantiates a field accessor for a property of a given POJO.
     * 
     * @param clazz
     *            The class for which to access the property
     * @param property
     *            The property name to access.
     * @throws NoSuchMethodException
     *             If no getter for the given property is found.
     * @throws SecurityException
     *             If the security context does not allow access to the getter.
     */
    @SuppressWarnings("unchecked")
    public POJOFieldAccessor(Class clazz, String property)
            throws NoSuchMethodException, SecurityException {
        super();
        this.field = null;

        String getterName = PropertyHelper.getGetterName(property);

        try {
            this.fieldGetter = clazz.getMethod(getterName);
        } catch (NoSuchMethodException e) {
            getterName = PropertyHelper.getBooleanGetterName(property);
            this.fieldGetter = clazz.getMethod(getterName);
        }

        if (this.fieldGetter.getReturnType().equals(Boolean.TYPE)) {
            if (!getterName.startsWith("is"))
                throw new NoSuchMethodException("Method [" + this.fieldGetter
                        + "] has a boolean return value.");
        } else {
            if (!getterName.startsWith("get"))
                throw new NoSuchMethodException("Method [" + this.fieldGetter
                        + "] has a non-boolean return value.");
        }

        if (Modifier.isStatic(this.fieldGetter.getModifiers()))
            throw new NoSuchMethodException("Method [" + this.fieldGetter
                    + "] is a static function.");

        String setterName = PropertyHelper.getSetterName(property);

        try {
            this.fieldSetter = this.fieldGetter.getDeclaringClass().getMethod(
                    setterName, this.fieldGetter.getReturnType());

            if (Modifier.isStatic(this.fieldSetter.getModifiers()))
                this.fieldSetter = null;

        } catch (SecurityException e) {
            this.fieldSetter = null;
        } catch (NoSuchMethodException e) {
            this.fieldSetter = null;
        }

        this.fieldName = property;
    }

    /**
     * Instantiates a field accessor for a property of a given POJO specified
     * through the supplied getter method.
     * 
     * @param getter
     *            The getter method to access the property
     * @throws NoSuchMethodException
     *             If the supplied method is not valid getter method.
     */
    public POJOFieldAccessor(Method getter) throws NoSuchMethodException {
        super();
        this.field = null;

        if (Modifier.isStatic(getter.getModifiers()))
            throw new NoSuchMethodException("Method [" + this.fieldGetter
                    + "] is a static function.");
        final Class retType = getter.getReturnType();
        if (retType.equals(Boolean.TYPE))
            this.fieldName = PropertyHelper
                    .getBooleanPropertyFromGetterName(getter.getName());
        else
            this.fieldName = PropertyHelper.getPropertyFromGetterName(getter
                    .getName());

        if (this.fieldName == null && retType.equals(Boolean.class))
            this.fieldName = PropertyHelper
                    .getBooleanPropertyFromGetterName(getter.getName());

        if (this.fieldName == null)
            throw new NoSuchMethodException("Method [" + getter
                    + "] is not a valid getter.");

        if (getter.getParameterTypes().length > 0)
            throw new NoSuchMethodException("Method [" + getter
                    + "] has one or more arguments.");

        this.fieldGetter = getter;

        String setterName = PropertyHelper.getSetterName(this.fieldName);

        try {
            this.fieldSetter = this.fieldGetter.getDeclaringClass().getMethod(
                    setterName, this.fieldGetter.getReturnType());

            if (Modifier.isStatic(this.fieldSetter.getModifiers()))
                this.fieldSetter = null;

        } catch (SecurityException e) {
            this.fieldSetter = null;
        } catch (NoSuchMethodException e) {
            this.fieldSetter = null;
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getValueOf(java.lang.Object)
     */
    public Object getValueOf(Object pojo) throws IllegalArgumentException,
            IllegalAccessException, InvocationTargetException {
        if (this.fieldGetter != null)
            return this.fieldGetter.invoke(pojo);
        else
            return this.field.get(pojo);
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#setValueOn(java.lang.Object,
     * java.lang.Object)
     */
    public void setValueOn(Object pojo, Object value)
            throws IllegalArgumentException, IllegalAccessException,
            InvocationTargetException {
        if (this.fieldSetter != null)
            this.fieldSetter.invoke(pojo, value);
        else if (this.field != null)
            this.field.set(pojo, value);
        else
            throw new IllegalAccessException("No setter for field ["
                    + this.fieldName + "] has been found.");
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getAnnotation(java.lang.Class)
     */
    public <T extends Annotation> T getAnnotation(Class<T> type) {
        T ann = null;
        if (this.field != null)
            ann = this.field.getAnnotation(type);
        if (ann == null && this.fieldGetter != null)
            ann = this.fieldGetter.getAnnotation(type);
        return ann;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getAnnotations()
     */
    public Annotation[] getAnnotations() {
        if (this.field != null)
            return this.field.getAnnotations();
        else
            return this.fieldGetter.getAnnotations();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getName()
     */
    public String getName() {
        return this.fieldName;
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getType()
     */
    public Class getType() {
        if (this.field != null)
            return this.field.getType();
        else
            return this.fieldGetter.getReturnType();
    }

    /*
     * (non-Javadoc)
     * 
     * @see com.efkon.validation.FieldAccessor#getDeclaringClass()
     */
    public Class getDeclaringClass() {
        if (this.field != null)
            return this.field.getDeclaringClass();
        else
            return this.fieldGetter.getDeclaringClass();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.util.reflect.FieldAccessor#checkMandatory()
     */
    public boolean checkMandatory() {
        if (getType().isPrimitive())
            return true;

        Column colAnn = getColumnAnnotation();
        if (colAnn != null)
            return (!colAnn.nullable());

        return false;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.util.reflect.FieldAccessor#checkUpdatable()
     */
    public boolean checkUpdatable() {

        Column colAnn = getColumnAnnotation();
        if (colAnn != null)
            return colAnn.updatable();

        return true;
    }

    /*
     * (non-Javadoc)
     * 
     * @see
     * org.clazzes.util.reflect.FieldAccessor#setIdAccessor(org.clazzes.util
     * .reflect.FieldAccessor)
     */
    public void setIdAccessor(FieldAccessor idAccessor) {
        this.idAccesor = idAccessor;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.clazzes.util.reflect.FieldAccessor#getIdAccessor()
     */
    public FieldAccessor getIdAccessor() {
        return this.idAccesor;
    }

    private Column getColumnAnnotation() {

        Column colAnn = getAnnotation(Column.class);
        if (colAnn != null)
            return colAnn;

        AttributeOverrides attributeOverrides = getAnnotation(AttributeOverrides.class);
        if (attributeOverrides != null) {
            AttributeOverride[] attributeOverride = attributeOverrides.value();
            if (attributeOverride.length > 0) {
                colAnn = attributeOverride[0].column();
                if (colAnn != null)
                    return colAnn;
            }
        }

        AttributeOverride attributeOverride = getAnnotation(AttributeOverride.class);
        if (attributeOverride != null) {
            colAnn = attributeOverride.column();
            if (colAnn != null)
                return colAnn;
        }
        return null;
    }

}
