/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.reflect;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;

/**
 * This interface is implemented by all classes, that allow access to fileds of
 * an object.
 * 
 * @author wglas
 */
public interface FieldAccessor {
    
    /**
     * @return The type of the field or property accessed.
     */
    public Class getType();

    /**
     * @return The class which contains the field or property accessed.
     */
    public Class getDeclaringClass();

    /**
     * @param type
     *            The annotation type to request for.
     * @return The annotation of the given field or property getter of the given
     *         type or null, if no such annotation exists.
     */
    public <T extends Annotation> T getAnnotation(Class<T> type);

    /**
     * @return All annotations of the given field or property getter or null, if
     *         no annotations exists.
     */
    public Annotation[] getAnnotations();

    /**
     * @return The name of the field or property accessed.
     */
    public String getName();

    /**
     * Gets the associated field value of the given instance of the underlying
     * class.
     * 
     * @param pojo
     *            The object of which to get the field value.
     * @return The value of the associated field.
     */
    public Object getValueOf(Object pojo) throws IllegalArgumentException,
            IllegalAccessException, InvocationTargetException;

    /**
     * Sets the associated field value on the given instance of the underlying
     * class.
     * 
     * @param pojo
     *            The object on which to set the field value.
     * @param value
     *            The value of the associated field.
     */
    public void setValueOn(Object pojo, Object value)
            throws IllegalArgumentException, IllegalAccessException,
            InvocationTargetException;

    /**
     * Parse all relevant Annotations and try to retrieve a <code>Column</code>
     * -Annotation and test it's <code>nullable()</code> value.
     * 
     * @return <code>true</code> for primitives,  
     *         <code>(!Column.nullable())</code> if a <code>Column</code>
     *         -Annotation could be found,  
     *         <code>false</code> otherwise.
     */
    public boolean checkMandatory();

    /**
     * Parse all relevant Annotations and try to retrieve a <code>Column</code>
     * -Annotation and test it's <code>updatable()</code> value.
     * 
     * @return <code>true</code> for primitives,  
     *         <code>Column.updatable()</code> if a <code>Column</code>
     *         -Annotation could be found,  
     *         <code>true</code> otherwise.
     */
    public boolean checkUpdatable();

    /**
     * Set the accessor used to access the ID-Field of the same POJO this
     * <code>FieldAccessor</code> belongs to.
     * 
     * @param idAccessor
     *            The <code>FieldAccessor</code> for the ID-Field.
     */
    public void setIdAccessor(FieldAccessor idAccessor);

    /**
     * Get the accessor used to access the ID-Field of the same POJO this
     * <code>FieldAccessor</code> belongs to.
     * 
     * @return The <code>FieldAccessor</code> for the ID-Field (may be null).
     */
    public FieldAccessor getIdAccessor();

}
