/***********************************************************
 * $Id$
 * 
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: Dec 27, 2006
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 ***********************************************************/

package org.clazzes.util.net;


/**
 * @author lech
 * 
 * This class provides a few IPv4 helper classes.
 * 
 * If anyone needs more IPv4 or IPv6 helper functions, please contact
 * christoph.lechleitner@iteg.at, who is maintaining and extending such
 * for the non-public MSrvMgr project.
 */
public class NetUtil {

    /**
     * Set to private, because this class defines no public non-static method.
     */
    private NetUtil() {
    }
    
    /**
     * Parses a string containing a dotted IPv4 address and converts it to a
     * long. Does no checks of any kind, use isValidIPv4 for that!
     * 
     * @param ip
     *            The string containing the IPv4 address, e.g. "1.2.3.4"
     * @return the long containing the address, e.g. 0x01020304
     */
    public static long ipString2Long(String ip) {
        String octets[] = ip.split("\\.");
        long nr = Long.valueOf(octets[0]).longValue() << 24
                | Long.valueOf(octets[1]).longValue() << 16
                | Long.valueOf(octets[2]).longValue() << 8
                | Long.valueOf(octets[3]).longValue();
        return nr;
    }

    /**
     * Take a long representing an IPv4 address and converts it to a printable
     * string
     * 
     * @param nr
     *            the long containing the address, e.g. 0x01020304
     * @return The string containing the IPv4 address, e.g. "1.2.3.4"
     */
    public static String ipLong2String(long nr) {
        String ip = String.valueOf((nr & 0xFF000000) >> 24) + "."
                + String.valueOf((nr & 0x00FF0000) >> 16) + "."
                + String.valueOf((nr & 0x0000FF00) >> 8) + "."
                + String.valueOf((nr & 0x000000FF));
        return ip;
    }

    /**
     * Checks if a string contains (only) a valid IPv4 address. Does not check
     * detail like zero subnets, only if syntax is ok and values are in range
     * 0-255.0-255.0-255.0-255
     * 
     * @param ip
     *            Suspect string to check
     * @return if the given string looks like a valid IPv4 address
     */
    public static boolean isValidIPv4(String ip) {
        if (!ip.matches("[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}\\.[0-9]{1,3}"))
            return false;
        String octets[] = ip.split("\\.");
        // the reason for those 4 ints is potential later advanced checks
        int i0 = Integer.valueOf(octets[0]);
        int i1 = Integer.valueOf(octets[1]);
        int i2 = Integer.valueOf(octets[2]);
        int i3 = Integer.valueOf(octets[3]);
        if (i0 < 0 || i0 > 255)
            return false;
        if (i1 < 0 || i1 > 255)
            return false;
        if (i2 < 0 || i2 > 255)
            return false;
        if (i3 < 0 || i3 > 255)
            return false;
        return true;
    }

}
