/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 2006-03-13
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.lang;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.zip.CRC32;

import org.apache.commons.codec.binary.Hex;
import org.clazzes.util.sec.HashTools;

public class Util {

    /**
     * Enumeration of all OS we are able to detect
     */
    public enum OperatingSystem {
        UNKOWN, WINDOWS, UNIX
    };

    public final static String CHARSET_ASCII = "US-ASCII";

    public final static String CHARSET_ISO88591 = "ISO-8859-1";

    public final static String CHARSET_UTF8 = "UTF-8";

    /**
     * OS we are working in. Use
     *
     * @getOperatingSystem() and
     * @inWindows() to avoid Unkown answers
     */
    private static OperatingSystem operating_system = OperatingSystem.UNKOWN;

    /**
     * Set to private, because this class defines no public non-static method.
     */
    private Util() {
    }

    /**
     * Compares a and b and returns true, if both values are null or
     * both are equal.
     *
     * @param a The first object to compare.
     * @param b The second object to compare.
     * @return <code>true</code>, if <code>a==null &amp;&amp; b==null</code>
     *         or <code>a.equals(b)</code>.
     */
    public static boolean equalsNullAware(Object a, Object b) {
        if (a == null)
            return b == null;
        return a.equals(b);
    }

    /**
     * Compares a and b and returns a number less than, equal to
     * or greater than zero, if <code>a</code> is greater than, equal to
     * or less than <code>b</code>.
     *
     * <code>null</code> values are considered
     * as the least possible value.
     *
     * @param a The first object to compare.
     * @param b The second object to compare.
     * @return A number less than, equal to or greater than zero,
     *         if <code>a</code> is greater than, equal to or less than <code>b</code>.
     */
    public static int compareNullAware(Comparable a, Comparable b)
    {
        if (a == null) {
            return b == null ? 0 : -1;
        }
        else if (b == null){

            return 1;
        }
        else {
            return a.compareTo(b);
        }
    }

    /**
     * Calculate a standard CRC32 Checksum using java.util.zip.CRC32.
     *
     * @param data
     *            The Bytes to calculate the Checksum from.
     * @return The Checksum
     */
    public static int getCRC32(final byte[] data) {

        CRC32 crc = new CRC32();
        crc.update(data);

        return (int) crc.getValue();
    }

    /**
     * Fills up a Byte Array to a multiple of x Bytes. fillUp(byte[12], 16,
     * 0x00) will produce a byte[16], the last 4 Bytes are 0x00.
     * fillUp(byte[16], 16, 0x13) will produce a copy of the byte[16].
     * fillUp(byte[18], 16, 0x01) will produce a byte[32], the lase 14 Bytes are
     * 0x01. fillUp(byte[97], 16, 0xFF) will produce a byte[112], the last 15
     * Bytes are 0xFF.
     *
     * @param src
     *            The Byte Array to fill up.
     * @param length
     *            The src ByteArray will be filled up to a length of n*length
     *            Bytes.
     * @param fillWith
     *            The value of the extra Bytes.
     * @return The filled up Byte Array.
     */
    public static byte[] fillUp(byte[] src, final int length,
            final byte fillWith) {
        int rest = src.length % length;
        if (rest == 0)
            return src;
        int newLength = src.length + length - rest;
        byte[] newSrc = new byte[newLength];
        System.arraycopy(src, 0, newSrc, 0, src.length);
        for (int i = src.length; i < newLength; i++)
            newSrc[i] = fillWith;
        return newSrc;
    }

    /**
     * Procudes an explicitely filled byte array with the given length. Useful
     * to produce "empty" arrays whose bytes are reliable set to 0.
     *
     * @param length
     *            The length of the byte array to be created.
     * @param fillWith
     *            The value of all bytes of the new byte array.
     * @return The filled Byte Array.
     */
    public static byte[] filledByteArray(final int length, final byte fillWith) {
        byte[] ba = new byte[length];
        for (int i = 0; i < length; i++) {
            ba[i] = fillWith;
        }
        return ba;
    }

    public static Calendar getUTCnow() {
        return Calendar.getInstance(TimeZone.getTimeZone("UTC"));
    }

    public static Calendar getUTCtimestamp(long millis) {
        Calendar cal = GregorianCalendar.getInstance(TimeZone
                .getTimeZone("UTC"));
        cal.setTimeInMillis(millis);
        return cal;
    }

    /**
     * Get a Calendar-Object for the given date (starting at 00:00:00.000.
     *
     * @param year
     *            The human readable year (2006)
     * @param month
     *            The human readable month (April = 4)
     * @param day
     *            The day of the month.
     * @return A calendar for the given date using UTC as the timezone and
     *         the time set to <code>00:00:00</code>
     */
    public static Calendar getCalendar(final int year, final int month,
            final int day) {
        Calendar cal = GregorianCalendar.getInstance(TimeZone
                .getTimeZone("UTC"));
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.MONTH, month - 1);
        cal.set(Calendar.DAY_OF_MONTH, day);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal;
    }

    /**
     * Calculate the XOR between two Byte arrays;
     *
     * @param a The first input byte array
     * @param b The second input byte array,
     *         which must have the same length than <code>a</code>.
     * @return A byte array, which contains <code>a[i]^b[i]</code> at
     *         component number <code>i</code>.
     *
     * @throws IllegalArgumentException When the lengths of the byte arrays do not match.
     */
    public static byte[] xor(final byte[] a, final byte[] b) {
        if (a == null || b == null) {
            throw new NullPointerException("arrays must not be null.");
        } else {
            if (a.length != b.length) {
                throw new IllegalArgumentException(
                        "arrays must be of the same length. (" + a.length
                                + " != " + b.length + ")");
            }
        }
        int len = a.length;
        byte[] buffer = new byte[a.length];
        for (int i = 0; i < len; i++) {
            int v1 = a[i] & 0xff;
            int v2 = b[i] & 0xff;
            int xor = v1 ^ v2;
            buffer[i] = (byte) (xor & 0xff);
        }
        return buffer;
    }

    /**
     * Invert a Byte array.
     *
     * @param a An input byte array.
     * @return A byte array of length <code>a.length</code> which contains
     *         <code>a[i]^0xff</code>.
     */
    public static byte[] invert(final byte[] a) {
        if (a == null) {
            throw new NullPointerException("Array must not be null.");
        }
        return (xor(a, filledByteArray(a.length, u8ToByte(0xFF))));
    }

    /**
     * Give me a SecureRandom of a desired length.
     *
     * @param len
     *            The Length of the random value.
     * @return The random value.
     */
    public static byte[] getRandom(final int len) {
        byte[] random = new byte[len];
        try {
            SecureRandom srnd = HashTools.getSecureRandom();
            srnd.nextBytes(random);
        } catch (Exception ex) {
            ex.printStackTrace();
        }

        return random;
    }

    /**
     * Give me the SHA1 Value of a Byte array.
     *
     * @param input The input data to use.
     * @return An SHA-1 hash of the given input data.
     * @throws Exception Upon errors.
     */
    public static byte[] getSHA1hash(final byte[] input) throws Exception {

        byte[] result = null;

        MessageDigest md = MessageDigest.getInstance("SHA1");
        md.update(input);
        result = md.digest();

        return result;
    }

    /**
     * Give me the SHA1 Value of a String. This method is deprecated, because
     * it uses the default encoding of the JVM.
     *
     * @param str The string to hash.
     * @return An SHA-1 hash of the given string in the standard encoding
     *         of the JVM.
     * @throws Exception Upon errors.
     */
    @Deprecated
    public static String getSHA1hash(final String str) throws Exception {

        MessageDigest md = MessageDigest.getInstance("SHA1");
        md.update(str.getBytes());
        String result = Util.asHex(md.digest(), "");

        return result;
    }

    /**
     * Test if two Byte arrays are equal. This function is deprecated and
     * should be replaced by {@link Arrays#equals(byte[], byte[])} wherever possible.
     * The only difference to <code>Arrays.equals()</code> is, that this method throws
     * a NPE when one of the arguments is <code>null</code>.
     *
     * @param a byte array one.
     * @param b byte array two.
     * @return If the two byte arrays have the same length an the same
     *         byte by byte content.
     * @throws NullPointerException If one of the two byte arrays is <code>null</code>.
     */
    @Deprecated
    public static boolean equals(final byte[] a, final byte[] b)
            throws NullPointerException {

        if (a == null || b == null) {
            throw new NullPointerException("arrays must not be null.");
        } else {
            return Arrays.equals(a,b);
        }
    }

    /**
     * Copy two Byte arrays into a single resulting Byte array.
     *
     * @param a byte array one.
     * @param b byte array two.
     * @return A byte array of length <code>a.length+b.length</code> containing the
     *         content of <code>a</code> and <code>b</code> concatenated in this order.
     */
    public static byte[] concat(final byte[] a, final byte[] b) {

        byte[] result = new byte[a.length + b.length];
        System.arraycopy(a, 0, result, 0, a.length);
        System.arraycopy(b, 0, result, a.length, b.length);

        return result;
    }

    /**
     * <p>Return a sub array of a given byte array.</p>
     *
     * <p>This method is deprecated, use
     * <code>Arrays.copyOfRange(src,offset,offset+length)</code>
     * instead. </p>
     *
     * @param src The source byte array.
     * @param offset The start offset.
     * @param length The length of the sub array.
     * @return A byte array containing the bytes <code>offset</code>
     *         to <code>offset+length</code>.
     */
    @Deprecated
    public static byte[] subArray(final byte[] src, final int offset,
            final int length) {

        byte[] result = new byte[length];
        System.arraycopy(src, offset, result, 0, length);

        return result;
    }

    /**
     * <p>Turns array of bytes into string.</p>
     *
     * @param buf Array of bytes to convert to hex string, with a space between
     *            each byte.
     * @return Generated hex string (without any 0x prefix).
     */
    public static String asHex(final byte buf[]) {
        return asHex(buf, " ");
    }

    /**
     * <p>Turns array of bytes into string.</p>
     *
     * <p>If you do not need a separator, {@link Hex#encode(byte[])} of <code>commons-codec</code>
     * is a much faster alternative compared to calling <code>Util.asHex(b,"")</code>.</p>
     *
     * @param buf
     *            Array of bytes to convert to hex string.
     * @param sep
     *            Separator to put between each byte, null stands for ""
     * @return Generated hex string (without any 0x prefix).
     */
    public static String asHex(final byte buf[], String sep) {
        if (buf == null)
            return "null";

        StringBuffer strbuf = new StringBuffer(buf.length * 2);
        int i;

        for (i = 0; i < buf.length; i++) {
            if (((int) buf[i] & 0xff) < 0x10)
                strbuf.append("0");

            strbuf.append(Long.toString((int) buf[i] & 0xff, 16));
            if (i < (buf.length - 1) && sep != null)
                strbuf.append(sep);
        }

        //log.debug(strbuf.toString());

        return strbuf.toString();
    }

    /**
     * <p>Decodes a string contain hex digits to a byte[].</p>
     *
     * <p>If you do not need to handle separators or an optional <code>0x</code> prefix,
     * {@link Hex#decodeHex(char[])} of <code>commons-codec</code>
     * is a much faster alternative compared to calling <code>Util.decodeHexstring(b,"")</code>.</p>
     * @param s
     *            The string to decode.
     * @return The decoded byte[]
     * @throws RuntimeException if the string was invalid
     */
    public static byte[] decodeHexstring(String s) {
        String s2 = s.replaceAll(" ", "").replaceFirst("0x", ""); // we want
                                                                    // accept
                                                                    // "0x 01 02
                                                                    // 03"
        int bal = s2.length() / 2;
        byte[] ba = new byte[bal];
        for (int i = 0; i < bal; i++) {
            String sub = s2.substring(i * 2, i * 2 + 2);
            ba[i] = u8ToByte(Short.parseShort(sub, 16));
        }
        return ba;
    }

    /**
     * Converts a (pseudo-signed) byte to an unsigned 8 bit value.
     *
     * @param b A byte.
     * @return <code>(int)b &amp; 0xff</code>
     */
    public static int byteToU8(byte b) {
        return (int)b & 0xff;
    }

    /**
     * Converts an unsigned 8 bit value to a (pseudo-signed) byte.
     * This function does the same as  the cast <code>(byte)u</code>, but may economize
     * typing and make the code more readable in some situations.
     *
     * @param u An unsigned integer.
     * @return u casted to a byte.
     */
    public static byte u8ToByte(int u) {
        return (byte)u;
    }

    /**
     * Converts a String object to an byte[] containing the ASCII coded string
     * and a 0 byte.
     *
     * @param s A string to convert.
     * @return A byte array with ASCII char codes and a terminating zero byte.
     */
    public static byte[] createZeroTerminatedString(String s) {
        return createZeroTerminatedString(s, CHARSET_ASCII);
    }

    /**
     * Converts a String object to an byte[] containing the string coded with
     * the given charset and a 0 byte.
     *
     * @param s The string to convert.
     * @param charset The ID of the charset to use.
     *            We recommend to use this class' CHARSET_* constants
     * @return A byte array with char codes in the given encoding and
     *         a terminating zero byte.
     */
    public static byte[] createZeroTerminatedString(String s, String charset) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream((s == null ? 0
                : s.length() + 1));
        try {
            if (s != null)
                baos.write(s.getBytes(charset));
        } catch (UnsupportedEncodingException e) {
        } catch (IOException e) {
        }
        baos.write((byte) 0);
        return baos.toByteArray();
    }

    /**
     * Converts a zero terminated ASCII string to a java String object,
     * suggesting ASCII charset.
     *
     * @param ba A byte array with ASCII char codes and a terminating zero byte.
     * @return A string representing the given ASCII input.
     */
    public static String parseZeroTerminatedString(byte[] ba) {
        return parseZeroTerminatedString(ba, CHARSET_ASCII);
    }

    /**
     * Converts a zero terminated string in the given encoding
     * to a java String object.
     *
     * @param ba A byte array with char codes in the given encoding
     *           and a terminating zero byte.
     * @param charset The ID of the charset to use.
     *            We recommend to use this class' CHARSET_* constants
     * @return A string representing the given encoded input.
     */
    public static String parseZeroTerminatedString(byte[] ba, String charset) {
        String s = null;
        try {
            if (ba != null && ba.length > 1) {
                // find 0
                int zi = 0;
                while (zi < ba.length) {
                    if (ba[zi] == 0)
                        break;
                    zi++;
                }
                // if 0 was not found, zi=ba.length+1, this is ok
                s = new String(ba, 0, zi, charset);
            }
        } catch (UnsupportedEncodingException e) {
        }
        return s;
    }

    /**
     * Returns which type of OS we have, after detecting if necessary
     *
     * @return The operating system guessed from the system property
     *         <code>os.name</code>
     */
    public static OperatingSystem getOperatingSystem() {
        if (operating_system == OperatingSystem.UNKOWN) {
            String osName = System.getProperty("os.name");
            if (osName.toLowerCase().contains("window")) // !windows
            {
                operating_system = OperatingSystem.WINDOWS;
            } else {
                operating_system = OperatingSystem.UNIX;
            }

        }
        return operating_system;
    }

    /**
     * Tells if we are under M$ Windows, after checking if necessary
     *
     * @return Whether {@link #getOperatingSystem()} returned {@link OperatingSystem#WINDOWS}
     */
    public static boolean inWindows() {
        return (getOperatingSystem() == OperatingSystem.UNKOWN);
    }

    /**
     * Computes an byte[2] containing the BCD representation of the given int
     *
     * @param pin
     *            The pin to encoded, 1 .. 9999
     * @return The BCD-encoded value of <code>pin</code>.
     */
    public static byte[] asBCD16(int pin) {
        byte[] ba = new byte[2];
        int bcd = 0;
        bcd |= (pin % 10);
        pin /= 10;
        bcd |= (pin % 10) << 4;
        pin /= 10;
        bcd |= (pin % 10) << 8;
        pin /= 10;
        bcd |= (pin % 10) << 12;
        ba[0] = (byte) ((bcd & 0xFF00) >> 8);
        ba[1] = (byte) ((bcd & 0x00FF));
        return ba;
    }

    /**
     * Decodes a BCD coded integer
     *
     * @param ba
     *            The bytearray containing the BCD bytes
     * @return The decoded value
     */
    public static int decodeBCD(byte[] ba) {
        int ret = 0;
        for (int i = 0; i < ba.length; i++) {
            ret *= 100;
            ret += ((ba[i] & 0xF0) >> 4) * 10 + (ba[i] & 0x0F);
        }
        return ret;
    }

    //
    // number of decimal digits of each power of 2.
    //
    // output of python code:
    // ",".join([str(math.floor(math.log10(2)*i)) for i in range(65)])
    private static final int[] DECIMAL_DIGITS_GUESS = new int[]
            {0,0,0,0,
             1,1,1,2,
             2,2,3,3,
             3,3,4,4,
             4,5,5,5,
             6,6,6,6,
             7,7,7,8,
             8,8,9,9,
             9,9,10,10,
             10,11,11,11,
             12,12,12,12,
             13,13,13,14,
             14,14,15,15,
             15,15,16,16,
             16,17,17,17,
             18,18,18,18,
             19};

    private static final int[] TEN_TO_THE_INTS = new int[] {
            0,10,100,1000,10000,100000,1000000,10000000,100000000,1000000000
    };

    private static final long[] TEN_TO_THE_LONGS = new long[] {
            0L,10L,100L,
            1000L,10000L,100000L,
            1000000L,10000000L,100000000L,
            1000000000L,10000000000L,100000000000L,
            1000000000000L,10000000000000L,100000000000000L,
            1000000000000000L,10000000000000000L,100000000000000000L,
            1000000000000000000L
    };

    /**
     * Calculate the length of the decimal representation of the given positive long value.
     * @param v A positive value.
     * @return The length of the decimal representation, which is 1 for zero.
     */
    public static int decimalLength(long v) {

        // see https://stackoverflow.com/questions/25892665
        int highBit = 64 - Long.numberOfLeadingZeros(v);

        int guess = DECIMAL_DIGITS_GUESS[highBit];

        return v >= TEN_TO_THE_LONGS[guess] ? guess+1: guess;
    }

    /**
     * Calculate the length of the decimal representation of the given positive int value.
     * @param v A positive value.
     * @return The length of the decimal representation, which is 1 for zero.
     */
    public static int decimalLength(int v) {

        // see https://stackoverflow.com/questions/25892665
        int highBit = 32 - Integer.numberOfLeadingZeros(v);

        int guess = DECIMAL_DIGITS_GUESS[highBit];

        return v >= TEN_TO_THE_INTS[guess] ? guess+1: guess;
    }
}
