/***********************************************************
 * $Id$
 *
 * Utility classes of the clazzes.org project
 * http://www.clazzes.org
 *
 * Created: 05.05.2022
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************/

package org.clazzes.util.lang;

import java.util.AbstractList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * A read-only view of a list that maps its elements using mapper.
 * Example:
 * <pre>{@code
 * Pair<Integer, String> pair = new Pair<Integer, String>(1, "test");
 * List<Pair<Integer, String>> list = new ArrayList<Pair<Integer, String>>();
 * list.add(pair);
 * List<Integer> mappedList = MappedList.newInstance(list, Pair::getFirst);
 * assert mappedList.get(0).equals(1);
 * }</pre>
 */
public class MappedList<P, T> extends AbstractList<T> {
    private List<P> underlying;
    private Function<P, T> mapper;

	/**
	 * Constructs a MappedList.
	 * @param underlying The underlying list.
	 * @param mapper The mapper.
	 */
    public MappedList(List<P> underlying, Function<P, T> mapper) {
        Objects.requireNonNull(underlying);
        Objects.requireNonNull(mapper);

        this.underlying = underlying;
        this.mapper = mapper;
    }

	/**
	 * Same as {@link #MappedList(List, Function)} with nicer type inference.
     */
    public static <P, T> MappedList<P, T> newInstance(List<P> underlying, Function<P, T> mapper) {
        return new MappedList<P, T>(underlying, mapper);
    };

    @Override
    public T get(int i) {
        return this.mapper.apply(this.underlying.get(i));
    }

    @Override
    public int size() {
        return this.underlying.size();
    }
}
